<?php
if ( ! defined( 'myCRED_VERSION' ) ) exit;

/**
 * myCRED_Hooks_Module class
 * @since 0.1
 * @version 1.3
 */
if ( ! class_exists( 'myCRED_Hooks_Module' ) ) :
	class myCRED_Hooks_Module extends myCRED_Module {

		public $setup;

		/**
		 * Construct
		 */
		function __construct( $type = MYCRED_DEFAULT_TYPE_KEY ) {

			parent::__construct( 'myCRED_Hooks_Module', array(
				'module_name' => 'hooks',
				'option_id'   => 'mycred_pref_hooks',
				'defaults'    => array(
					'installed'   => array(),
					'active'      => array(),
					'hook_prefs'  => array()
				),
				'labels'      => array(
					'menu'        => __( 'Hooks', 'mycred' ),
					'page_title'  => __( 'Hooks', 'mycred' ),
					'page_header' => __( 'Hooks', 'mycred' )
				),
				'screen_id'   => MYCRED_SLUG . '-hooks',
				'accordion'   => false,
				'menu_pos'    => 20
			), $type );

		}

		/**
		 * Load Hooks
		 * @since 0.1
		 * @version 1.1
		 */
		public function module_init() {

			// Loop through each active hook and call the run() method.
			if ( ! empty( $this->installed ) ) {

				foreach ( $this->installed as $key => $gdata ) {

					if ( $this->is_active( $key ) && isset( $gdata['callback'] ) ) {
						$this->call( 'run', $gdata['callback'] );
					}

				}

			}

			// Ajax handlers for hook management
			add_action( 'wp_ajax_mycred-hook-order',  array( $this, 'ajax_hook_activation' ) );
			add_action( 'wp_ajax_mycred-save-hook',   array( $this, 'ajax_save_hook_prefs' ) );

		}

		/**
		 * Get Hooks
		 * @since 0.1
		 * @version 1.2
		 */
		public function get( $save = false ) {

			$installed = array();

			// Registrations
			$installed['registration'] = array(
				'title'         => __( '%plural% for registrations', 'mycred' ),
				'description'   => __( 'Award %_plural% for users joining your website.', 'mycred' ),
				'documentation' => 'http://codex.mycred.me/hooks/registrations/',
				'callback'      => array( 'myCRED_Hook_Registration' )
			);

			// Site Visits
			$installed['site_visit'] = array(
				'title'         => __( '%plural% for daily visits', 'mycred' ),
				'description'   => __( 'Award %_plural% for visiting your website on a daily basis.', 'mycred' ),
				'documentation' => 'http://codex.mycred.me/hooks/daily-visits/',
				'callback'      => array( 'myCRED_Hook_Site_Visits' )
			);

			// View Content
			$installed['view_contents'] = array(
				'title'         => __( '%plural% for viewing content', 'mycred' ),
				'description'   => __( 'Award %_plural% for viewing content.', 'mycred' ),
				'documentation' => 'http://codex.mycred.me/hooks/viewing-content/',
				'callback'      => array( 'myCRED_Hook_View_Contents' )
			);

			// Logins
			$installed['logging_in'] = array(
				'title'         => __( '%plural% for logins', 'mycred' ),
				'description'   => __( 'Award %_plural% for logging in.', 'mycred' ),
				'documentation' => 'http://codex.mycred.me/hooks/logins/',
				'callback'      => array( 'myCRED_Hook_Logging_In' )
			);

			// Content Publishing
			$installed['publishing_content'] = array(
				'title'         => __( '%plural% for publishing content', 'mycred' ),
				'description'   => __( 'Award %_plural% for publishing content.', 'mycred' ),
				'documentation' => 'http://codex.mycred.me/hooks/publishing-content/',
				'callback'      => array( 'myCRED_Hook_Publishing_Content' )
			);

			// Content Deletions
			$installed['deleted_content'] = array(
				'title'         => __( '%plural% for trashed content', 'mycred' ),
				'description'   => __( 'Award or Deduct %_plural% when content gets trashed.', 'mycred' ),
				'documentation' => 'http://codex.mycred.me/hooks/deleting-content/',
				'callback'      => array( 'myCRED_Hook_Delete_Content' )
			);

			// Commenting
			$installed['comments'] = array(
				'title'         => ( ! function_exists( 'dsq_is_installed' ) ) ? __( '%plural% for comments', 'mycred' ) : __( '%plural% for Disqus comments', 'mycred' ),
				'description'   => __( 'Award %_plural% for making comments.', 'mycred' ),
				'documentation' => 'http://codex.mycred.me/hooks/comments/',
				'callback'      => array( 'myCRED_Hook_Comments' )
			);

			// Link Clicks
			$installed['link_click'] = array(
				'title'         => __( '%plural% for clicking on links', 'mycred' ),
				'description'   => str_replace( '%shortcode%', '<a href="http://codex.mycred.me/shortcodes/mycred_link/" target="_blank">mycred_link</a>', __( 'Award %_plural% for clicks on links generated by the %shortcode% shortcode.', 'mycred' ) ),
				'documentation' => 'http://codex.mycred.me/hooks/clicking-on-links/',
				'callback'      => array( 'myCRED_Hook_Click_Links' )
			);

			// Video Views
			$installed['video_view'] = array(
				'title'         => __( '%plural% for viewing Videos', 'mycred' ),
				'description'   => str_replace( '%shortcode%', '<a href="http://codex.mycred.me/shortcodes/mycred_video/" target="_blank">mycred_video</a>', __( 'Award %_plural% for watches videos embedded using the %shortcode% shortcode.', 'mycred' ) ),
				'documentation' => 'http://codex.mycred.me/hooks/watching-videos/',
				'callback'      => array( 'myCRED_Hook_Video_Views' )
			);

			// Affiliation
			$installed['affiliate'] = array(
				'title'         => __( '%plural% for referrals', 'mycred' ),
				'description'   => __( 'Award %_plural% for signup or visitor referrals.', 'mycred' ),
				'documentation' => 'http://codex.mycred.me/hooks/referrals/',
				'callback'      => array( 'myCRED_Hook_Affiliate' )
			);

			$installed = apply_filters( 'mycred_setup_hooks', $installed, $this->mycred_type );

			if ( $save === true && $this->core->can_edit_plugin() ) {
				$new_data = array(
					'active'     => $this->active,
					'installed'  => $installed,
					'hook_prefs' => $this->hook_prefs
				);
				mycred_update_option( $this->option_id, $new_data );
			}

			$this->installed = $installed;
			return $installed;

		}

		/**
		 * Call
		 * Either calls a given class method or function.
		 * @since 0.1
		 * @version 1.1.1
		 */
		public function call( $call, $callback, $return = NULL ) {

			// Class
			if ( is_array( $callback ) && class_exists( $callback[0] ) ) {

				$class = $callback[0];
				$methods = get_class_methods( $class );
				if ( in_array( $call, $methods ) ) {

					$new = new $class( ( isset( $this->hook_prefs ) ) ? $this->hook_prefs : array(), $this->mycred_type );
					return $new->$call( $return );

				}

			}

			// Function
			elseif ( ! is_array( $callback ) ) {

				if ( function_exists( $callback ) ) {

					if ( $return !== NULL )
						return call_user_func( $callback, $return, $this );
					else
						return call_user_func( $callback, $this );

				}

			}

			if ( $return !== NULL )
				return array();

		}

		/**
		 * Settings Header
		 * @since 1.7
		 * @version 1.0
		 */
		function settings_header() {

			wp_enqueue_style( 'mycred-bootstrap-grid' );
			wp_enqueue_style( 'mycred-forms' );

			wp_localize_script(
				'mycred-widgets',
				'myCREDHooks',
				array(
					'type' => $this->mycred_type
				)
			);
			wp_enqueue_script( 'mycred-widgets' );

			if ( wp_is_mobile() )
				wp_enqueue_script( 'jquery-touch-punch' );

		}

		/**
		 * Admin Page
		 * @since 0.1
		 * @version 1.2.1
		 */
		public function admin_page() {

			// Security
			if ( ! $this->core->can_edit_creds() ) wp_die( 'Access Denied' );

			// Get installed
			$installed   = $this->get();
			$this->setup = mycred_get_option( $this->option_id . '_sidebar', 'default' );

			$button = '';
			// For later - right now this feature is disabled until I can come up with an easy to imlement this
			// $button      = '<a href="' . add_query_arg( array( 'page' => $_GET['page'], 'switch-hook' => 'default', 'token' => wp_create_nonce( 'mycred-switch-hook-role' ) ), admin_url( 'admin.php' ) ) . '" class="page-title-action hide-if-no-customize">' . __( 'One for all', 'mycred' ) . '</a>';
			// if ( $this->setup == 'default' )
			// 	$button  = '<a href="' . add_query_arg( array( 'page' => $_GET['page'], 'switch-hook' => 'role', 'token' => wp_create_nonce( 'mycred-switch-hook-default' ) ), admin_url( 'admin.php' ) ) . '" class="page-title-action hide-if-no-customize">' . __( 'Role Based', 'mycred' ) . '</a>';

?>
<style type="text/css">
.widget-content { display: block; float: none; clear: both; }
.widget-content label.subheader { display: block; font-weight: bold; padding: 0 0 0 0; margin: 0 0 6px 0; }
.widget-content ol { margin: 0 0 6px 0; }
.widget-content ol.inline:after { content: ""; display: block; height: 1px; clear: both; }
.widget-content ol li { list-style-type: none; margin: 0 0 0 0; padding: 0 0 0 0; }
.widget-content ol.inline li { display: block; float: left; min-width: 45%; }
.widget-content ol.inline li.empty { display: none; }
.widget-content ol li input, .widget-content ol li select { margin-bottom: 6px; }
.widget-content ol li input[type="checkbox"], .widget-content ol li input[type="radio"] { margin-bottom: 0; }
.widget-content ol li input.mini { margin-right: 12px; }
.widget-content ol li input.long { width: 100%; }
.widget-content ol li label { display: block; margin-bottom: 6px; }
.widget-content select.limit-toggle { vertical-align: top; }

.widget-content .hook-instance { margin-bottom: 18px; border-bottom: 1px dashed #d5d5d5; }
.widget-content .hook-instance:last-of-type { border-bottom: none; margin-bottom: 0; }
.widget-content .hook-instance h3 { margin: 0 0 12px 0; }
.widget-content .hook-instance .row > div .form-group:last-child { margin-bottom: 0; }
.widget-content .page-title-action { top: 0; float: right; }

#sidebar-active .widget-inside .form .form-group span.description { display: block; font-style: italic; font-size: 12px; line-height: 16px; padding-left: 0; padding-right: 0; padding-top: 6px; }
#available-widgets .widget .widget-description { min-height: 50px; }
#sidebar-active .widget-inside form .widget-content { padding-top: 12px; }
#sidebar-active .widget-inside form .widget-control-actions { padding-top: 12px; border-top: 1px dashed #dedede; margin-top: 12px; }
.form .radio { margin-bottom: 12px; }
</style>
<div class="wrap">
	<h1><?php echo sprintf( __( '%s Hooks %s', 'mycred' ), mycred_label(), $button ); ?> <a href="http://codex.mycred.me/chapter-ii/setup-hooks/" class="page-title-action" target="_blank"><?php _e( 'Documentation', 'mycred' ); ?></a></h1>
	<div class="widget-liquid-left">
		<div id="widgets-left">
			<div id="available-widgets" class="widgets-holder-wrap">
				<div class="sidebar-name">
					<div class="sidebar-name-arrow"><br /></div>
					<h2><?php _e( 'Available Hooks' ); ?> <span id="removing-widget"><?php _ex( 'Deactivate', 'removing-widget' ); ?> <span></span></span></h2>
				</div>
				<div class="widget-holder">
					<div class="sidebar-description">
						<p class="description"><?php _e( 'To activate a hook drag it to a sidebar or click on it. To deactivate a hook and delete its settings, drag it back.' ); ?></p>
					</div>
					<div id="widget-list">
<?php

			// If we have hooks
			if ( ! empty( $installed ) ) {

				global $mycred_field_id;

				$mycred_field_id = '__i__';

				// Loop though them
				$count = 0;
				foreach ( $installed as $key => $data ) {

?>
						<div id="widget-mycred-hook_<?php echo $key; ?>" class="widget ui-draggable"<?php if ( $this->is_active( $key ) ) echo ' style="display: none;"'; ?>>
							<div class="widget-top">
								<div class="widget-title-action"></div>
								<div class="widget-title ui-draggable-handle">
									<h3><?php echo $this->core->template_tags_general( $data['title'] ); ?></h3>
								</div>
							</div>
							<div class="widget-inside mycred-metabox">
								<form method="post" action="" class="form">
									<div class="widget-content">

										<?php $this->call( 'preferences', $data['callback'] ); ?>

									</div>
									<input type="hidden" name="widget-id" class="widget-id" value="<?php echo $key; ?>" />
									<input type="hidden" name="id_base" class="id_base" value="<?php echo $key; ?>" />
									<input type="hidden" name="add_new" class="add_new" value="single" />
									<div class="widget-control-actions">
										<div class="alignleft">
											<a class="widget-control-remove" href="#remove"><?php _e( 'Delete', 'mycred' ); ?></a> | <a class="widget-control-close" href="#close"><?php _e( 'Close', 'mycred' ); ?></a><?php if ( array_key_exists( 'documentation', $data ) && ! empty( $data['documentation'] ) ) : ?> | <a class="hook-documentation" href="<?php echo esc_url( $data['documentation'] ); ?>" target="_blank">Hook Documentation</a><?php endif; ?>
										</div>
										<div class="alignright">
											<input type="submit" name="savewidget" id="widget-mycred-hook-<?php echo $key; ?>-__i__-savewidget" class="button button-primary widget-control-save right" value="<?php _e( 'Save', 'mycred' ); ?>" />
											<span class="spinner"></span>
										</div>
										<br class="clear" />
									</div>
								</form>
							</div>
							<div class="widget-description"><?php echo nl2br( $this->core->template_tags_general( $data['description'] ) ); ?></div>
						</div>
<?php

					$count++;
				}

				$mycred_field_id = '';

			}

?>
					</div>
					<br class="clear" />
				</div>
				<br class="clear" />
			</div>
		</div>
	</div>
	<div class="widget-liquid-right">

		<?php $this->display_sidebars(); ?>

	</div>
	<form method="post"><?php wp_nonce_field( 'manage-mycred-hooks', '_wpnonce_widgets', false ); ?></form>
	<br class="clear" />
</div>
<div class="widgets-chooser">
	<ul class="widgets-chooser-sidebars"></ul>
	<div class="widgets-chooser-actions">
		<button class="button-secondary"><?php _e( 'Cancel', 'mycred' ); ?></button>
		<button class="button-primary"><?php _e( 'Add Hook', 'mycred' ); ?></button>
	</div>
</div>

<script type="text/javascript">
jQuery(function($) {

	$( 'div.widget-liquid-right' ).on( 'change', 'select.limit-toggle', function(){

		if ( $(this).find( ':selected' ).val() != 'x' )
			$(this).prev().attr( 'type', 'text' ).val( 0 );
		else
			$(this).prev().attr( 'type', 'hidden' ).val( 0 );

	});

});
</script>
<?php

		}

		/**
		 * Display Sidebars
		 * @since 1.7
		 * @version 1.0
		 */
		public function display_sidebars() {

			// Default setup
			if ( $this->setup == 'default' ) {

?>
<div id="widgets-right" class="single-sidebar">
	<div class="sidebars-column-0">
		<div class="widgets-holder-wrap">
			<div id="sidebar-active" class="widgets-sortables ui-droppable ui-sortable">
				<div class="sidebar-name">
					<div class="sidebar-name-arrow"><br /></div>
					<h2><?php _e( 'Active Hooks', 'mycred' ); ?></h2>
				</div>
				<div class="sidebar-description">
					<p class="description"><?php _e( 'The following hooks are used for all users.', 'mycred' ); ?></p>
				</div>
<?php

			// If we have hooks
			if ( ! empty( $this->installed ) ) {

				// Loop though them
				foreach ( $this->installed as $key => $data ) {

					// Show only active hooks
					if ( ! $this->is_active( $key ) ) continue;

?>
				<div id="widget-mycred-hook_<?php echo $key; ?>" class="widget" style="z-index: auto;">
					<div class="widget-top">
						<div class="widget-title-action"></div>
						<div class="widget-title ui-draggable-handle">
							<h3><?php echo $this->core->template_tags_general( $data['title'] ); ?></h3>
						</div>
					</div>
					<div class="widget-inside mycred-metabox">
						<form method="post" action="" class="form">
							<div class="widget-content">

								<?php $this->call( 'preferences', $data['callback'] ); ?>

							</div>
							<input type="hidden" name="widget-id" class="widget-id" value="<?php echo $key; ?>" />
							<input type="hidden" name="id_base" class="id_base" value="<?php echo $key; ?>" />
							<input type="hidden" name="add_new" class="add_new" value="single" />
							<div class="widget-control-actions">
								<div class="alignleft">
									<a class="widget-control-remove" href="#remove"><?php _e( 'Delete', 'mycred' ); ?></a> | <a class="widget-control-close" href="#close"><?php _e( 'Close', 'mycred' ); ?></a><?php if ( array_key_exists( 'documentation', $data ) && ! empty( $data['documentation'] ) ) : ?>  | <a class="hook-documentation" href="<?php echo esc_url( $data['documentation'] ); ?>" target="_blank">Hook Documentation</a><?php endif; ?>
								</div>
								<div class="alignright">
									<input type="submit" name="savewidget" id="widget-mycred-hook-<?php echo $key; ?>-__i__-savewidget" class="button button-primary widget-control-save right" value="<?php _e( 'Save', 'mycred' ); ?>" />
									<span class="spinner"></span>
								</div>
								<br class="clear" />
							</div>
						</form>
					</div>
					<div class="widget-description"><?php echo nl2br( $this->core->template_tags_general( $data['description'] ) ); ?></div>
				</div>
<?php

				}

			}

?>

			</div>
		</div>
	</div>
</div>
<?php

			}

			// Let others play
			else {

				do_action( 'mycred-hook-sidebars' , $this );
				do_action( 'mycred-hook-sidebars-' . $this->mycred_type , $this );

			}

		}

		/**
		 * AJAX: Save Hook Activations
		 * Either saves the hook order (no use) or saves hooks being activated or deactivated.
		 * @since 1.7
		 * @version 1.0.1
		 */
		public function ajax_hook_activation() {

			check_ajax_referer( 'manage-mycred-hooks', 'savewidgets' );

			if ( ! isset( $_POST['sidebars'] ) ) die;

			$ctype      = sanitize_key( $_POST['ctype'] );
			if ( $ctype !== $this->mycred_type ) return;

			$installed = $this->get();

			if ( ! empty( $_POST['sidebars'] ) ) {
				foreach ( $_POST['sidebars'] as $sidebar_id => $hooks ) {

					$hooks = explode( ',', $hooks );

					// First get all the hook IDs
					$clean_hook_ids = array();
					if ( ! empty( $hooks ) ) {
						foreach ( $hooks as $hook_id ) {
							$clean_hook_ids[] = sanitize_key( str_replace( array( 'new-widget-mycred-hook_', 'widget-mycred-hook_' ), '', $hook_id ) );
						}
					}

					// One for all
					if ( $sidebar_id == 'sidebar-active' ) {

						$active_hooks = array();
						if ( ! empty( $this->active ) && ! empty( $clean_hook_ids ) ) {
							foreach ( $this->active as $already_active_hook_id ) {

								// Retain active hooks that are set to remain active
								if ( in_array( $already_active_hook_id, $clean_hook_ids ) && ! in_array( $already_active_hook_id, $active_hooks ) )
									$active_hooks[] = $already_active_hook_id;

							}
						}

						// Loop through all hooks in this sidebase and consider them as active
						if ( ! empty( $clean_hook_ids ) ) {
							foreach ( $clean_hook_ids as $hook_id ) {

								if ( array_key_exists( $hook_id, $installed ) && ! in_array( $hook_id, $active_hooks ) )
									$active_hooks[] = $hook_id;

							}
						}

						$active_hooks = array_unique( $active_hooks, SORT_STRING );
						$this->active = $active_hooks;

						// Update our settings to activate the hook(s)
						mycred_update_option( $this->option_id, array(
							'active'     => $this->active,
							'installed'  => $installed,
							'hook_prefs' => $this->hook_prefs
						) );

					}

					// Role based
					else {
						
					}

				}
			}

		}

		/**
		 * AJAX: Save Hook Settings
		 * @since 1.7
		 * @version 1.0.3
		 */
		public function ajax_save_hook_prefs() {

			check_ajax_referer( 'manage-mycred-hooks', 'savewidgets' );

			$sidebar    = sanitize_text_field( $_POST['sidebar'] );
			$hook_id    = sanitize_key( $_POST['id_base'] );
			$ctype      = sanitize_key( $_POST['ctype'] );
			$hook_prefs = false;

			if ( $ctype !== $this->mycred_type ) return;

			// Get hook settings
			if ( $ctype == MYCRED_DEFAULT_TYPE_KEY && array_key_exists( $hook_id, $_POST['mycred_pref_hooks']['hook_prefs'] ) )
				$hook_prefs = $_POST['mycred_pref_hooks']['hook_prefs'][ $hook_id ];

			elseif ( $ctype != MYCRED_DEFAULT_TYPE_KEY && array_key_exists( $hook_id, $_POST[ 'mycred_pref_hooks_' . $ctype ]['hook_prefs'] ) )
				$hook_prefs = $_POST[ 'mycred_pref_hooks_' . $ctype ]['hook_prefs'][ $hook_id ];

			if ( $hook_prefs === false ) die;

			$installed = $this->get();
			if ( ! array_key_exists( $hook_id, $installed ) )
				die( '<p>No longer available hook</p>' );

			// New settings
			$new_settings = $this->call( 'sanitise_preferences', $installed[ $hook_id ]['callback'], $hook_prefs );

			// If something went wrong use the old settings
			if ( ! is_array( $new_settings ) || empty( $new_settings ) )
				$new_settings = $hook_prefs;

			$this->hook_prefs[ $hook_id ] = $new_settings;

			// Update our settings to activate the hook(s)
			mycred_update_option( $this->option_id, array(
				'active'     => $this->active,
				'installed'  => $installed,
				'hook_prefs' => $this->hook_prefs
			) );

			$this->call( 'preferences', $installed[ $hook_id ]['callback'] );
			die;

		}

	}
endif;

/**
 * Hook for registrations
 * @since 0.1
 * @version 1.2
 */
if ( ! class_exists( 'myCRED_Hook_Registration' ) ) :
	class myCRED_Hook_Registration extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			parent::__construct( array(
				'id'       => 'registration',
				'defaults' => array(
					'creds'   => 10,
					'log'     => '%plural% for becoming a member'
				)
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 0.1
		 * @version 1.2.1
		 */
		public function run() {

			add_action( 'user_register', array( $this, 'registration' ) );

		}

		/**
		 * Registration Hook
		 * @since 0.1
		 * @version 1.1
		 */
		public function registration( $user_id ) {

			// Make sure user is not excluded
			if ( $this->core->exclude_user( $user_id ) === true ) return;

			$data = array( 'ref_type' => 'user' );

			// Make sure this is unique
			if ( $this->core->has_entry( 'registration', $user_id, $user_id, $data, $this->mycred_type ) ) return;

			// Execute
			$this->core->add_creds(
				'registration',
				$user_id,
				$this->prefs['creds'],
				$this->prefs['log'],
				$user_id,
				$data,
				$this->mycred_type
			);

		}

		/**
		 * Preference for Registration Hook
		 * @since 0.1
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<div class="row">
		<div class="col-lg-4 col-md-4 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'creds' ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'creds' ); ?>" id="<?php echo $this->field_id( 'creds' ); ?>" value="<?php echo $this->core->number( $prefs['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-8 col-md-8 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'log' ); ?>"><?php _e( 'Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'log' ); ?>" id="<?php echo $this->field_id( 'log' ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'user' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

		}

	}
endif;

/**
 * Hook for site visits
 * @since 1.5
 * @version 1.1
 */
if ( ! class_exists( 'myCRED_Hook_Site_Visits' ) ) :
	class myCRED_Hook_Site_Visits extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			parent::__construct( array(
				'id'       => 'site_visit',
				'defaults' => array(
					'creds'   => 1,
					'log'     => '%plural% for site visit'
				)
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 1.5
		 * @version 1.0.2
		 */
		public function run() {

			// Make sure user is logged in. Also to prevent unneccery db queries we
			// check to make sure the user does not have the cookie.
			if ( is_user_logged_in() && ! isset( $_COOKIE['mycred_site_visit'] ) )
				add_action( 'wp_head', array( $this, 'site_visit' ) );

		}

		/**
		 * Visit Hook
		 * @since 1.5
		 * @version 1.1.3
		 */
		public function site_visit() {

			if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) return;

			// Current User ID
			$user_id = get_current_user_id();
			$now     = current_time( 'timestamp' );

			// Set cookie to prevent db queries again today.
			$lifespan = (int) ( 24*3600 ) - ( date( 'H', $now ) * 3600 + date( 'i', $now ) * 60 + date( 's', $now ) );
			if ( ! headers_sent() ) setcookie( 'mycred_site_visit', 1, $lifespan, COOKIEPATH, COOKIE_DOMAIN );

			// Make sure user is not excluded
			if ( $this->core->exclude_user( $user_id ) ) return;

			// Store todays date as an integer
			$today = (int) apply_filters( 'mycred_site_visit_id', date( 'Ymd', $now ) );
			$data = '';

			// Make sure this is unique
			if ( $this->core->has_entry( 'site_visit', $today, $user_id, $data, $this->mycred_type ) ) return;

			// Execute
			$this->core->add_creds(
				'site_visit',
				$user_id,
				$this->prefs['creds'],
				$this->prefs['log'],
				$today,
				$data,
				$this->mycred_type
			);

		}

		/**
		 * Preference for Site Visit Hook
		 * @since 1.5
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<div class="row">
		<div class="col-lg-4 col-md-4 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'creds' ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'creds' ); ?>" id="<?php echo $this->field_id( 'creds' ); ?>" value="<?php echo $this->core->number( $prefs['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-8 col-md-8 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'log' ); ?>"><?php _e( 'Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'log' ); ?>" id="<?php echo $this->field_id( 'log' ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

		}

	}
endif;

/**
 * Hook for viewing content
 * @since 1.5.1
 * @version 1.2
 */
if ( ! class_exists( 'myCRED_Hook_View_Contents' ) ) :
	class myCRED_Hook_View_Contents extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			$defaults = array(
				'post'    => array(
					'creds'   => 1,
					'log'     => '%plural% for viewing a post',
					'acreds'  => 1,
					'limit'   => '0/x',
					'alog'    => '%plural% for view of your post',
					'visitor' => 0,
					'alimit'  => '0/x'
				),
				'page'    => array(
					'creds'   => 1,
					'log'     => '%plural% for viewing a page',
					'acreds'  => 1,
					'limit'   => '0/x',
					'alog'    => '%plural% for view of your page',
					'visitor' => 0,
					'alimit'  => '0/x'
				)
			);

			if ( isset( $hook_prefs['view_contents'] ) )
				$defaults = $hook_prefs['view_contents'];

			parent::__construct( array(
				'id'       => 'view_contents',
				'defaults' => $defaults
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 1.5.1
		 * @version 1.0.1
		 */
		public function run() {

			// First instance where we can safely use conditional template tags
			add_action( 'template_redirect', array( $this, 'content_loading' ), 999 );

		}

		/**
		 * Content Loaded
		 * @since 1.5.1
		 * @version 1.2
		 */
		public function content_loading() {

			// Only applicable on single post type view by logged in users
			if ( ! is_singular() || ! is_user_logged_in() ) return;

			global $post;

			$user_id    = get_current_user_id();
			$pay_author = true;
			$data       = array( 'ref_type' => 'post' );

			// Post author can not generate points for themselves
			if ( $post->post_author == $user_id ) return;

			// Make sure this post type award points. Any amount but zero.
			if ( isset( $this->prefs[ $post->post_type ]['creds'] ) && $this->prefs[ $post->post_type ]['creds'] != 0 && apply_filters( 'mycred_view_content', true, $this ) === true ) {

				// Make sure we are not excluded
				if ( ! $this->core->exclude_user( $user_id ) ) {

					// Enforce limit and make sure users only get points once per unique post
					if ( ! $this->over_hook_limit( $post->post_type, 'view_content', $user_id ) && ! $this->core->has_entry( 'view_content', $post->ID, $user_id, $data, $this->mycred_type ) ) {

						$this->core->add_creds(
							'view_content',
							$user_id,
							$this->prefs[ $post->post_type ]['creds'],
							$this->prefs[ $post->post_type ]['log'],
							$post->ID,
							$data,
							$this->mycred_type
						);

					}

					// If the visitor does not get points, neither does the author
					else $pay_author = false;

				}

			}

			// Make sure this post type award points to the author. Any amount but zero.
			if ( isset( $this->prefs[ $post->post_type ]['acreds'] ) && $this->prefs[ $post->post_type ]['acreds'] != 0 && apply_filters( 'mycred_view_content_author', $pay_author, $this ) === true ) {

				// No payout for viewing our own content
				if ( ! $this->core->exclude_user( $post->post_author ) ) {

					$data['cui'] = $user_id;

					// Limit
					if ( ! $this->over_hook_limit( $post->post_type, 'view_content_author', $post->post_author ) )
						$this->core->add_creds(
							'view_content_author',
							$post->post_author,
							$this->prefs[ $post->post_type ]['acreds'],
							$this->prefs[ $post->post_type ]['alog'],
							$post->ID,
							$data,
							$this->mycred_type
						);

				}

			}

		}

		/**
		 * Preference for read content hook
		 * @since 1.5.1
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<h3><?php _e( 'Viewing Posts', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'creds' ) ); ?>"><?php _e( 'Member', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'post' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'post' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['post']['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post', 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'post', 'limit' ) ), $this->field_id( array( 'post', 'limit' ) ), $prefs['post']['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-2 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'acreds' ) ); ?>"><?php _e( 'Content Author', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'post' => 'acreds' ) ); ?>" id="<?php echo $this->field_id( array( 'post' => 'acreds' ) ); ?>" value="<?php echo $this->core->number( $prefs['post']['acreds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post', 'alimit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'post', 'alimit' ) ), $this->field_id( array( 'post', 'alimit' ) ), $prefs['post']['alimit'] ); ?>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'log' ) ); ?>"><?php _e( 'Member Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'post' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'post' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['post']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'alog' ) ); ?>"><?php _e( 'Content Author Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'post' => 'alog' ) ); ?>" id="<?php echo $this->field_id( array( 'post' => 'alog' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['post']['alog'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<div class="hook-instance">
	<h3><?php _e( 'Viewing Pages', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'creds' ) ); ?>"><?php _e( 'Member', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'page' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'page' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['page']['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page', 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'page', 'limit' ) ), $this->field_id( array( 'page', 'limit' ) ), $prefs['page']['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-2 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'acreds' ) ); ?>"><?php _e( 'Content Author', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'page' => 'acreds' ) ); ?>" id="<?php echo $this->field_id( array( 'page' => 'acreds' ) ); ?>" value="<?php echo $this->core->number( $prefs['page']['acreds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page', 'alimit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'page', 'alimit' ) ), $this->field_id( array( 'page', 'alimit' ) ), $prefs['page']['alimit'] ); ?>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'log' ) ); ?>"><?php _e( 'Member Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'page' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'page' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['page']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'alog' ) ); ?>"><?php _e( 'Content Author Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'page' => 'alog' ) ); ?>" id="<?php echo $this->field_id( array( 'page' => 'alog' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['page']['alog'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

			// Get all not built-in post types (excludes posts, pages, media)
			$post_type_args = array(
				'public'   => true,
				'_builtin' => false
			);
			$post_types = get_post_types( $post_type_args, 'objects', 'and' ); 

			foreach ( $post_types as $post_type ) {

				// Points to award/deduct
				if ( isset( $prefs[ $post_type->name ]['creds'] ) )
					$_creds = $prefs[ $post_type->name ]['creds'];
				else
					$_creds = 0;

				if ( isset( $prefs[ $post_type->name ]['limit'] ) )
					$limit = $prefs[ $post_type->name ]['limit'];
				else
					$limit = '0/x';

				// Log template
				if ( isset( $prefs[ $post_type->name ]['log'] ) )
					$_log = $prefs[ $post_type->name ]['log'];
				else
					$_log = '%plural% for viewing ' . $post_type->labels->name;

				// Points to award/deduct
				if ( isset( $prefs[ $post_type->name ]['acreds'] ) )
					$_acreds = $prefs[ $post_type->name ]['acreds'];
				else
					$_acreds = 0;

				if ( isset( $prefs[ $post_type->name ]['alimit'] ) )
					$alimit = $prefs[ $post_type->name ]['alimit'];
				else
					$alimit = '0/x';

				// Log template
				if ( isset( $prefs[ $post_type->name ]['alog'] ) )
					$_alog = $prefs[ $post_type->name ]['alog'];
				else
					$_alog = '%plural% for view of your ' . $post_type->labels->name;

?>
<div class="hook-instance">
	<h3><?php printf( __( 'Viewing %s', 'mycred' ), $post_type->labels->name ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'creds' ) ); ?>"><?php _e( 'Member', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( $post_type->name => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( $post_type->name => 'creds' ) ); ?>" value="<?php echo $this->core->number( $_creds ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name, 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( $post_type->name, 'limit' ) ), $this->field_id( array( $post_type->name, 'limit' ) ), $limit ); ?>
			</div>
		</div>
		<div class="col-lg-2 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'acreds' ) ); ?>"><?php _e( 'Content Author', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( $post_type->name => 'acreds' ) ); ?>" id="<?php echo $this->field_id( array( $post_type->name => 'acreds' ) ); ?>" value="<?php echo $this->core->number( $_acreds ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name, 'alimit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( $post_type->name, 'alimit' ) ), $this->field_id( array( $post_type->name, 'alimit' ) ), $alimit ); ?>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'log' ) ); ?>"><?php _e( 'Member', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( $post_type->name => 'log' ) ); ?>" id="<?php echo $this->field_id( array( $post_type->name => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $_log ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'alog' ) ); ?>"><?php _e( 'Content Author', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( $post_type->name => 'alog' ) ); ?>" id="<?php echo $this->field_id( array( $post_type->name => 'alog' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $_alog ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

			}

		}

		/**
		 * Sanitise Preferences
		 * @since 1.6
		 * @version 1.0.1
		 */
		function sanitise_preferences( $data ) {

			if ( isset( $data['post']['limit'] ) && isset( $data['post']['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['post']['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['post']['limit'] = $limit . '/' . $data['post']['limit_by'];
				unset( $data['post']['limit_by'] );
			}

			if ( isset( $data['post']['alimit'] ) && isset( $data['post']['alimit_by'] ) ) {
				$limit = sanitize_text_field( $data['post']['alimit'] );
				if ( $limit == '' ) $limit = 0;
				$data['post']['alimit'] = $limit . '/' . $data['post']['alimit_by'];
				unset( $data['post']['alimit_by'] );
			}

			if ( isset( $data['page']['limit'] ) && isset( $data['page']['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['page']['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['page']['limit'] = $limit . '/' . $data['page']['limit_by'];
				unset( $data['page']['limit_by'] );
			}

			if ( isset( $data['page']['alimit'] ) && isset( $data['page']['alimit_by'] ) ) {
				$limit = sanitize_text_field( $data['page']['alimit'] );
				if ( $limit == '' ) $limit = 0;
				$data['page']['alimit'] = $limit . '/' . $data['page']['alimit_by'];
				unset( $data['page']['alimit_by'] );
			}

			$post_type_args = array(
				'public'   => true,
				'_builtin' => false
			);
			$post_types = get_post_types( $post_type_args, 'objects', 'and' ); 

			foreach ( $post_types as $post_type ) {

				if ( isset( $data[ $post_type->name ]['limit'] ) && isset( $data[ $post_type->name ]['limit_by'] ) ) {
					$limit = sanitize_text_field( $data[ $post_type->name ]['limit'] );
					if ( $limit == '' ) $limit = 0;
					$data[ $post_type->name ]['limit'] = $limit . '/' . $data[ $post_type->name ]['limit_by'];
					unset( $data[ $post_type->name ]['limit_by'] );
				}

				if ( isset( $data[ $post_type->name ]['alimit'] ) && isset( $data[ $post_type->name ]['alimit_by'] ) ) {
					$limit = sanitize_text_field( $data[ $post_type->name ]['alimit'] );
					if ( $limit == '' ) $limit = 0;
					$data[ $post_type->name ]['alimit'] = $limit . '/' . $data[ $post_type->name ]['alimit_by'];
					unset( $data[ $post_type->name ]['alimit_by'] );
				}

			}

			return $data;

		}

	}
endif;

/**
 * Hook for loggins
 * @since 0.1
 * @version 1.1
 */
if ( ! class_exists( 'myCRED_Hook_Logging_In' ) ) :
	class myCRED_Hook_Logging_In extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			parent::__construct( array(
				'id'       => 'logging_in',
				'defaults' => array(
					'creds'   => 1,
					'log'     => '%plural% for logging in',
					'limit'   => '1/d'
				)
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 0.1
		 * @version 1.1
		 */
		public function run() {

			// Social Connect
			if ( function_exists( 'sc_social_connect_process_login' ) )
				add_action( 'social_connect_login', array( $this, 'social_login' ) );

			// WordPress
			add_action( 'wp_login', array( $this, 'logging_in' ), 10, 2 );

		}

		/**
		 * Social Login
		 * Adds support for Social Connect plugin
		 * @since 1.4
		 * @version 1.1
		 */
		public function social_login( $user_login = 0 ) {

			// Get user
			$user = get_user_by( 'login', $user_login );
			if ( ! isset( $user->ID ) ) {
				// In case we use emails for login instead of username
				$user = get_user_by( 'email', $user_login );
				if ( ! is_object( $user ) ) return;
			}

			// Check for exclusion
			if ( $this->core->exclude_user( $user->ID ) ) return;

			// Limit
			if ( ! $this->over_hook_limit( '', 'logging_in', $user->ID ) )
				$this->core->add_creds(
					'logging_in',
					$user->ID,
					$this->prefs['creds'],
					$this->prefs['log'],
					0,
					'',
					$this->mycred_type
				);

		}

		/**
		 * Login Hook
		 * @since 0.1
		 * @version 1.3
		 */
		public function logging_in( $user_login, $user = '' ) {

			// In case the user object is not past along
			if ( ! is_object( $user ) ) {

				$user = get_user_by( 'login', $user_login );
				if ( ! is_object( $user ) ) {

					// In case we use emails for login instead of username
					$user = get_user_by( 'email', $user_login );
					if ( ! is_object( $user ) ) return;

				}

			}

			// Check for exclusion
			if ( $this->core->exclude_user( $user->ID ) ) return;

			// Limit
			if ( ! $this->over_hook_limit( '', 'logging_in', $user->ID ) )
				$this->core->add_creds(
					'logging_in',
					$user->ID,
					$this->prefs['creds'],
					$this->prefs['log'],
					0,
					'',
					$this->mycred_type
				);

		}

		/**
		 * Preference for Login Hook
		 * @since 0.1
		 * @version 1.2
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'creds' ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'creds' ); ?>" id="<?php echo $this->field_id( 'creds' ); ?>" value="<?php echo $this->core->number( $prefs['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'limit' ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( 'limit' ), $this->field_id( 'limit' ), $prefs['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'log' ); ?>"><?php _e( 'Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'log' ); ?>" id="<?php echo $this->field_id( 'log' ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

		}

		/**
		 * Sanitise Preferences
		 * @since 1.6
		 * @version 1.0
		 */
		function sanitise_preferences( $data ) {

			if ( isset( $data['limit'] ) && isset( $data['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['limit'] = $limit . '/' . $data['limit_by'];
				unset( $data['limit_by'] );
			}

			return $data;

		}

	}
endif;

/**
 * Hook for publishing content
 * @since 0.1
 * @version 1.2
 */
if ( ! class_exists( 'myCRED_Hook_Publishing_Content' ) ) :
	class myCRED_Hook_Publishing_Content extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			$defaults = array(
				'post'    => array(
					'creds'  => 1,
					'log'    => '%plural% for new Post',
					'limit'  => '0/x'
				),
				'page'    => array(
					'creds'  => 1,
					'log'    => '%plural% for new Page',
					'limit'  => '0/x'
				)
			);

			if ( isset( $hook_prefs['publishing_content'] ) )
				$defaults = $hook_prefs['publishing_content'];

			parent::__construct( array(
				'id'       => 'publishing_content',
				'defaults' => $defaults
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 0.1
		 * @version 1.0
		 */
		public function run() {

			add_action( 'transition_post_status', array( $this, 'publishing_content' ), 10, 3 );

		}

		/**
		 * Publish Content Hook
		 * @since 0.1
		 * @version 1.1.3
		 */
		public function publishing_content( $new_status, $old_status, $post ) {

			$user_id   = $post->post_author;
			$post_id   = $post->ID;
			$post_type = $post->post_type;

			// Check for exclusions
			if ( $this->core->exclude_user( $user_id ) === true ) return;

			// Make sure we award points other then zero
			if ( ! isset( $this->prefs[ $post_type ]['creds'] ) || empty( $this->prefs[ $post_type ]['creds'] ) || $this->prefs[ $post_type ]['creds'] == 0 ) return;

			// We want to fire when content get published or when it gets privatly published
			$status         = apply_filters( 'mycred_publish_hook_old', array( 'new', 'auto-draft', 'draft', 'private', 'pending', 'future' ), $post_id, $this->mycred_type );
			$publish_status = apply_filters( 'mycred_publish_hook_new', array( 'publish', 'private' ), $post_id, $this->mycred_type );

			if ( in_array( $old_status, $status ) && in_array( $new_status, $publish_status ) && array_key_exists( $post_type, $this->prefs ) ) {

				// Prep
				$entry = $this->prefs[ $post_type ]['log'];
				$data  = array( 'ref_type' => 'post' );

				// Make sure this is unique
				if ( $this->core->has_entry( 'publishing_content', $post_id, $user_id, $data, $this->mycred_type ) ) return;

				// Check limit
				if ( ! $this->over_hook_limit( $post_type, 'publishing_content', $user_id ) )
					$this->core->add_creds(
						'publishing_content',
						$user_id,
						$this->prefs[ $post_type ]['creds'],
						$entry,
						$post_id,
						$data,
						$this->mycred_type
					);

			}

		}

		/**
		 * Preference for Publish Content Hook
		 * @since 0.1
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<h3><?php _e( 'Publishing Posts', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'creds' ) ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'post' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'post' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['post']['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'post' => 'limit' ) ), $this->field_id( array( 'post' => 'limit' ) ), $prefs['post']['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'post' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'post' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['post']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<div class="hook-instance">
	<h3><?php _e( 'Publishing Pages', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'creds' ) ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'page' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'page' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['page']['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'page' => 'limit' ) ), $this->field_id( array( 'page' => 'limit' ) ), $prefs['page']['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'page' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'page' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['page']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

			// Get all not built-in post types (excludes posts, pages, media)
			$post_type_args = array(
				'public'   => true,
				'_builtin' => false
			);
			$post_types     = get_post_types( $post_type_args, 'objects', 'and' ); 

			foreach ( $post_types as $post_type ) {

				// Start by checking if this post type should be excluded
				if ( ! $this->include_post_type( $post_type->name ) ) continue;

				// Points to award/deduct
				if ( isset( $prefs[ $post_type->name ]['creds'] ) )
					$_creds = $prefs[ $post_type->name ]['creds'];
				else
					$_creds = 0;

				// Log template
				if ( isset( $prefs[ $post_type->name ]['log'] ) )
					$_log = $prefs[ $post_type->name ]['log'];
				else
					$_log = '%plural% for published content';

				if ( isset( $prefs[ $post_type->name ]['limit'] ) )
					$_limit = $prefs[ $post_type->name ]['limit'];
				else
					$_limit = '0/x';

?>
<div class="hook-instance">
	<h3><?php printf( __( 'Publishing %s', 'mycred' ), $post_type->labels->name ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'creds' ) ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( $post_type->name => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( $post_type->name => 'creds' ) ); ?>" value="<?php echo $this->core->number( $_creds ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( $post_type->name => 'limit' ) ), $this->field_id( array( $post_type->name => 'limit' ) ), $_limit ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( $post_type->name => 'log' ) ); ?>" id="<?php echo $this->field_id( array( $post_type->name => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $_log ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

			}

		}

		/**
		 * Sanitise Preferences
		 * @since 1.6
		 * @version 1.0
		 */
		function sanitise_preferences( $data ) {

			if ( isset( $data['post']['limit'] ) && isset( $data['post']['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['post']['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['post']['limit'] = $limit . '/' . $data['post']['limit_by'];
				unset( $data['post']['limit_by'] );
			}

			if ( isset( $data['page']['limit'] ) && isset( $data['page']['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['page']['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['page']['limit'] = $limit . '/' . $data['page']['limit_by'];
				unset( $data['page']['limit_by'] );
			}

			// Get all not built-in post types (excludes posts, pages, media)
			$post_type_args = array(
				'public'   => true,
				'_builtin' => false
			);
			$post_types = get_post_types( $post_type_args, 'objects', 'and' ); 

			foreach ( $post_types as $post_type ) {

				// Start by checking if this post type should be excluded
				if ( ! $this->include_post_type( $post_type->name ) ) continue;

				if ( isset( $data[ $post_type->name ]['limit'] ) && isset( $data[ $post_type->name ]['limit_by'] ) ) {
					$limit = sanitize_text_field( $data[ $post_type->name ]['limit'] );
					if ( $limit == '' ) $limit = 0;
					$data[ $post_type->name ]['limit'] = $limit . '/' . $data[ $post_type->name ]['limit_by'];
					unset( $data[ $post_type->name ]['limit_by'] );
				}

			}

			return $data;

		}

	}
endif;

/**
 * Hook for deleting content
 * @since 1.7
 * @version 1.0
 */
if ( ! class_exists( 'myCRED_Hook_Delete_Content' ) ) :
	class myCRED_Hook_Delete_Content extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			$defaults = array(
				'post'    => array(
					'creds'  => 1,
					'log'    => '%plural% for deleted Post',
					'limit'  => '0/x'
				),
				'page'    => array(
					'creds'  => 1,
					'log'    => '%plural% for deleted Page',
					'limit'  => '0/x'
				)
			);

			if ( isset( $hook_prefs['deleted_content'] ) )
				$defaults = $hook_prefs['deleted_content'];

			parent::__construct( array(
				'id'       => 'deleted_content',
				'defaults' => $defaults
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 1.7
		 * @version 1.0
		 */
		public function run() {

			if ( EMPTY_TRASH_DAYS > 0 )
				add_action( 'trashed_post', array( $this, 'delete_content' ) );
			else
				add_action( 'before_delete_post', array( $this, 'delete_content' ) );

		}

		/**
		 * Delete Content Hook
		 * @since 1.7
		 * @version 1.0
		 */
		public function delete_content( $post_id ) {

			global $post_type;

			$post      = get_post( $post_id );

			$user_id   = $post->post_author;
			$post_type = $post->post_type;

			// Check for exclusions
			if ( $this->core->exclude_user( $user_id ) === true ) return;

			// Make sure we award points other then zero
			if ( ! isset( $this->prefs[ $post_type ]['creds'] ) || empty( $this->prefs[ $post_type ]['creds'] ) || $this->prefs[ $post_type ]['creds'] == 0 ) return;

			// Prep
			$entry = $this->prefs[ $post_type ]['log'];
			$data = array( 'ref_type' => 'post' );

			// Make sure this is unique
			if ( $this->core->has_entry( 'deleted_content', $post_id, $user_id, $data, $this->mycred_type ) ) return;

			// Check limit
			if ( ! $this->over_hook_limit( $post_type, 'deleted_content', $user_id ) )
				$this->core->add_creds(
					'deleted_content',
					$user_id,
					$this->prefs[ $post_type ]['creds'],
					$entry,
					$post_id,
					$data,
					$this->mycred_type
				);

		}

		/**
		 * Preference for Delete Content Hook
		 * @since 1.7
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<h3><?php _e( 'Trashing Posts', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'creds' ) ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'post' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'post' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['post']['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'post' => 'limit' ) ), $this->field_id( array( 'post' => 'limit' ) ), $prefs['post']['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'post' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'post' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'post' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['post']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<div class="hook-instance">
	<h3><?php _e( 'Trashing Pages', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'creds' ) ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'page' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'page' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['page']['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'page' => 'limit' ) ), $this->field_id( array( 'page' => 'limit' ) ), $prefs['page']['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'page' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'page' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'page' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['page']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

			// Get all not built-in post types (excludes posts, pages, media)
			$post_type_args = array(
				'public'   => true,
				'_builtin' => false
			);
			$post_types = get_post_types( $post_type_args, 'objects', 'and' ); 

			foreach ( $post_types as $post_type ) {

				// Start by checking if this post type should be excluded
				if ( ! $this->include_post_type( $post_type->name ) ) continue;

				// Points to award/deduct
				if ( isset( $prefs[ $post_type->name ]['creds'] ) )
					$_creds = $prefs[ $post_type->name ]['creds'];
				else
					$_creds = 0;

				// Log template
				if ( isset( $prefs[ $post_type->name ]['log'] ) )
					$_log = $prefs[ $post_type->name ]['log'];
				else
					$_log = '%plural% for deleted content';

				if ( isset( $prefs[ $post_type->name ]['limit'] ) )
					$_limit = $prefs[ $post_type->name ]['limit'];
				else
					$_limit = '0/x';

?>
<div class="hook-instance">
	<h3><?php printf( __( 'Trashing %s', 'mycred' ), $post_type->labels->name ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'creds' ) ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( $post_type->name => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( $post_type->name => 'creds' ) ); ?>" value="<?php echo $this->core->number( $_creds ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( $post_type->name => 'limit' ) ), $this->field_id( array( $post_type->name => 'limit' ) ), $_limit ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( $post_type->name => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( $post_type->name => 'log' ) ); ?>" id="<?php echo $this->field_id( array( $post_type->name => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $_log ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'post' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

			}

		}

		/**
		 * Sanitise Preferences
		 * @since 1.7
		 * @version 1.0
		 */
		function sanitise_preferences( $data ) {

			if ( isset( $data['post']['limit'] ) && isset( $data['post']['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['post']['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['post']['limit'] = $limit . '/' . $data['post']['limit_by'];
				unset( $data['post']['limit_by'] );
			}

			if ( isset( $data['page']['limit'] ) && isset( $data['page']['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['page']['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['page']['limit'] = $limit . '/' . $data['page']['limit_by'];
				unset( $data['page']['limit_by'] );
			}

			// Get all not built-in post types (excludes posts, pages, media)
			$post_type_args = array(
				'public'   => true,
				'_builtin' => false
			);
			$post_types = get_post_types( $post_type_args, 'objects', 'and' ); 

			foreach ( $post_types as $post_type ) {

				// Start by checking if this post type should be excluded
				if ( ! $this->include_post_type( $post_type->name ) ) continue;

				if ( isset( $data[ $post_type->name ]['limit'] ) && isset( $data[ $post_type->name ]['limit_by'] ) ) {
					$limit = sanitize_text_field( $data[ $post_type->name ]['limit'] );
					if ( $limit == '' ) $limit = 0;
					$data[ $post_type->name ]['limit'] = $limit . '/' . $data[ $post_type->name ]['limit_by'];
					unset( $data[ $post_type->name ]['limit_by'] );
				}

			}

			return $data;

		}

	}
endif;

/**
 * Hook for comments
 * @since 0.1
 * @version 1.3
 */
if ( ! class_exists( 'myCRED_Hook_Comments' ) ) :
	class myCRED_Hook_Comments extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			parent::__construct( array(
				'id'       => 'comments',
				'defaults' => array(
					'limits'   => array(
						'self_reply' => 0,
						'per_post'   => 10,
						'per_day'    => 0
					),
					'approved' => array(
						'creds'   => 1,
						'log'     => '%plural% for Approved Comment',
						'author'  => 0
					),
					'spam'     => array(
						'creds'   => '-5',
						'log'     => '%plural% deduction for Comment marked as SPAM',
						'author'  => 0
					),
					'trash'    => array(
						'creds'   => '-1',
						'log'     => '%plural% deduction for deleted / unapproved Comment',
						'author'  => 0
					)
				)
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 0.1
		 * @version 1.2
		 */
		public function run() {

			if ( ! function_exists( 'dsq_is_installed' ) ) {
				add_action( 'comment_post',              array( $this, 'new_comment' ), 99, 2 );
				add_action( 'transition_comment_status', array( $this, 'comment_transitions' ), 99, 3 );
			}
			else {
				add_action( 'wp_insert_comment',         array( $this, 'disqus' ), 99, 2 );
			}

		}

		/**
		 * New Comment
		 * If comments are approved without moderation, we apply the corresponding method
		 * or else we will wait till the appropriate instance.
		 * @since 0.1
		 * @version 1.2.2
		 */
		public function new_comment( $comment_id, $comment_status ) {

			// Marked SPAM
			if ( $comment_status === 'spam' )
				$this->comment_transitions( 'spam', 'unapproved', $comment_id );

			// Approved comment
			elseif ( $comment_status == 1 )
				$this->comment_transitions( 'approved', 'unapproved', $comment_id );

		}

		/**
		 * Discuss Support
		 * @since 1.4
		 * @version 1.0
		 */
		function disqus( $id, $comment ) {

			// Attempt to get a comment authors ID
			if ( $comment->user_id == 0 ) {

				$email = get_user_by( 'email', $comment->comment_author_email );
				// Failed to find author, can not award points
				if ( $email === false ) return;
				$comment->user_id = $email->ID;

			}

			$new_status = 'spam';
			if ( $comment->comment_approved == 1 )
				$new_status = 'approved';

			elseif ( $comment->comment_approved == 0 )
				$new_status = 'unapproved';

			$this->comment_transitions( $new_status, 'unapproved', $comment );

		}

		/**
		 * Comment Transitions
		 * @since 1.1.2
		 * @version 1.5
		 */
		public function comment_transitions( $new_status, $old_status, $comment ) {

			// Passing an integer instead of an object means we need to grab the comment object ourselves
			if ( ! is_object( $comment ) )
				$comment = get_comment( $comment );

			// No comment object so lets bail
			if ( $comment === NULL ) return;

			// Ignore Pingbacks or Trackbacks
			if ( $comment->comment_type != '' ) return;

			// Logged out users miss out
			if ( $comment->user_id == 0 ) return;

			if ( apply_filters( 'mycred_comment_gets_cred', true, $comment, $new_status, $old_status ) === false ) return;

			// Get comment author
			$comment_author = $comment->user_id;

			// Get content author
			$content_author = NULL;
			if ( isset( $comment->comment_post_ID ) || $comment->comment_post_ID != '' ) {
				$post = get_post( (int) $comment->comment_post_ID );
				$content_author = $post->post_author;
			}

			$comment_author_points = $this->core->zero();
			$content_author_points = $this->core->zero();

			$reference = '';
			$log = '';

			// Approved Comments
			if ( $new_status == 'approved' ) {
				$reference = 'approved_comment';
				$log = $this->prefs['approved']['log'];

				// From unapproved / hold
				if ( in_array( $old_status, array( 'unapproved', 'hold' ) ) ) {
					// Comment author
					if ( ! $this->user_exceeds_limit( $comment_author, $comment->comment_post_ID ) )
						$comment_author_points = $this->prefs['approved']['creds'];

					// Content author
					$content_author_points = $this->prefs['approved']['author'];
				}

				// From being marked as spam
				elseif ( $old_status == 'spam' ) {
					$comment_author_points = abs( $this->prefs['spam']['creds'] );
					$content_author_points = abs( $this->prefs['spam']['author'] );
				}

				// From being in trash
				elseif ( $old_status == 'trash' ) {
					$comment_author_points = abs( $this->prefs['trash']['creds'] );
					$content_author_points = abs( $this->prefs['trash']['author'] );
				}
			}

			// Unapproved Comments
			elseif ( $new_status == 'unapproved' && $old_status == 'approved' ) {
				$reference = 'unapproved_comment';
				$log = $this->prefs['trash']['log'];

				// If we deducted points for approved comments we want to add them back
				if ( $this->prefs['approved']['creds'] < $this->core->zero() ) {
					$comment_author_points = abs( $this->prefs['approved']['creds'] );
					$content_author_points = abs( $this->prefs['approved']['author'] );
				}

				// Else use what we have set
				else {
					$comment_author_points = $this->prefs['trash']['creds'];
					$content_author_points = $this->prefs['trash']['author'];
				}
			}

			// Marked as SPAM
			elseif ( $new_status == 'spam' ) {
				$reference = 'spam_comment';
				$log = $this->prefs['spam']['log'];

				$comment_author_points = $this->prefs['spam']['creds'];
				$content_author_points = $this->prefs['spam']['author'];
			}

			// Trashed Comments
			elseif ( $new_status == 'trash' ) {
				$reference = 'deleted_comment';
				$log = $this->prefs['trash']['log'];

				$comment_author_points = $this->prefs['trash']['creds'];
				$content_author_points = $this->prefs['trash']['author'];
			}

			// Comment Author
			if ( ! $this->core->exclude_user( $comment_author ) && $comment_author_points != $this->core->zero() ) {

				// Check if we are allowed to comment our own comment and are doing it
				if ( $this->prefs['limits']['self_reply'] != 0 && $comment->comment_parent != 0 ) {
					$parent = get_comment( $comment->comment_parent );
					// Comment author is not replying to their own comments
					if ( $parent->user_id != $comment_author ) {
						$this->core->add_creds(
							$reference,
							$comment_author,
							$comment_author_points,
							$log,
							$comment->comment_ID,
							array( 'ref_type' => 'comment' ),
							$this->mycred_type
						);
					}
				}
				// Else
				else {
					$this->core->add_creds(
						$reference,
						$comment_author,
						$comment_author_points,
						$log,
						$comment->comment_ID,
						array( 'ref_type' => 'comment' ),
						$this->mycred_type
					);
				}

			}

			if ( $content_author === NULL ) return;

			// Content Author
			if ( ! $this->core->exclude_user( $content_author ) && $content_author_points != $this->core->zero() ) {
				$this->core->add_creds(
					$reference,
					$content_author,
					$content_author_points,
					$log,
					$comment->comment_ID,
					array( 'ref_type' => 'comment' ),
					$this->mycred_type
				);
			}

		}

		/**
		 * Check if user exceeds limit
		 * @since 1.1.1
		 * @version 1.1
		 */
		public function user_exceeds_limit( $user_id = NULL, $post_id = NULL ) {

			if ( ! isset( $this->prefs['limits'] ) ) return false;

			// Prep
			$today = date( 'Y-m-d', current_time( 'timestamp' ) );

			// First we check post limit
			if ( $this->prefs['limits']['per_post'] > 0 ) {
				$post_limit = 0;

				// Grab limit
				if ( ! $this->is_main_type )
					$limit = mycred_get_user_meta( $user_id, 'mycred_comment_limit_post_' . $this->mycred_type, '', true );
				else
					$limit = mycred_get_user_meta( $user_id, 'mycred_comment_limit_post', '', true );

				// Apply default if none exist
				if ( empty( $limit ) ) $limit = array( $post_id => $post_limit );

				// Check if post_id is in limit array
				if ( array_key_exists( $post_id, $limit ) ) {
					$post_limit = $limit[ $post_id ];

					// Limit is reached
					if ( $post_limit >= $this->prefs['limits']['per_post'] ) return true;
				}

				// Add / Replace post_id counter with an incremented value
				$limit[ $post_id ] = $post_limit+1;

				// Save
				if ( ! $this->is_main_type )
					mycred_update_user_meta( $user_id, 'mycred_comment_limit_post_' . $this->mycred_type, '', $limit );
				else
					mycred_update_user_meta( $user_id, 'mycred_comment_limit_post', '', $limit );

			}

			// Second we check daily limit
			if ( $this->prefs['limits']['per_day'] > 0 ) {
				$daily_limit = 0;

				// Grab limit
				if ( ! $this->is_main_type )
					$limit = mycred_get_user_meta( $user_id, 'mycred_comment_limit_day_' . $this->mycred_type, '', true );
				else
					$limit = mycred_get_user_meta( $user_id, 'mycred_comment_limit_day', '', true );

				// Apply default if none exist
				if ( empty( $limit ) ) $limit = array();

				// Check if todays date is in limit
				if ( array_key_exists( $today, $limit ) ) {
					$daily_limit = $limit[ $today ];

					// Limit is reached
					if ( $daily_limit >= $this->prefs['limits']['per_day'] ) return true;
				}
				// Today is not in limit array so we reset to remove other dates
				else {
					$limit = array();
				}

				// Add / Replace todays counter with an imcremented value
				$limit[ $today ] = $daily_limit+1;

				// Save
				if ( ! $this->is_main_type )
					mycred_update_user_meta( $user_id, 'mycred_comment_limit_day_' . $this->mycred_type, '', $limit );
				else
					mycred_update_user_meta( $user_id, 'mycred_comment_limit_day', '', $limit );

			}

			return false;

		}

		/**
		 * Preferences for Commenting Hook
		 * @since 0.1
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

			if ( ! isset( $prefs['limits'] ) )
				$prefs['limits'] = array(
					'self_reply' => 0,
					'per_post'   => 10,
					'per_day'    => 0
				);

			if ( function_exists( 'dsq_is_installed' ) )
				echo '<p>' . $this->core->template_tags_general( __( '%plural% are only awarded when your website has been synced with the Disqus server!', 'mycred' ) ) . '</p>';

?>
<div class="hook-instance">
	<h3><?php _e( 'Approved Comments', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-3 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'approved' => 'creds' ) ); ?>"><?php _e( 'Member', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'approved' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'approved' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['approved']['creds'] ); ?>" class="form-control" />
				<span class="description"><?php _e( 'Use zero to disable.', 'mycred' ); ?></span>
			</div>
		</div>
		<div class="col-lg-3 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'approved' => 'author' ) ); ?>"><?php _e( 'Content Author', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'approved' => 'author' ) ); ?>" id="<?php echo $this->field_id( array( 'approved' => 'author' ) ); ?>" value="<?php echo $this->core->number( $prefs['approved']['author'] ); ?>" class="form-control" />
				<span class="description"><?php _e( 'Use zero to disable.', 'mycred' ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'approved' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'approved' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'approved' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['approved']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'comment' ) ); ?></span>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-12 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<div class="checkbox">
					<label for="<?php echo $this->field_id( array( 'limits' => 'self_reply' ) ); ?>"><input type="checkbox" name="<?php echo $this->field_name( array( 'limits' => 'self_reply' ) ); ?>" id="<?php echo $this->field_id( array( 'limits' => 'self_reply' ) ); ?>" <?php checked( $prefs['limits']['self_reply'], 1 ); ?> value="1" /> <?php echo $this->core->template_tags_general( __( '%plural% is to be awarded even when comment authors reply to their own comment.', 'mycred' ) ); ?></label>
				</div>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'limits' => 'per_post' ) ); ?>"><?php _e( 'Limit per post', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'limits' => 'per_post' ) ); ?>" id="<?php echo $this->field_id( array( 'limits' => 'per_post' ) ); ?>" value="<?php echo esc_attr( $prefs['limits']['per_post'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->core->template_tags_general( __( 'The number of comments per post that grants %_plural% to the comment author. Use zero for unlimited.', 'mycred' ) ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'limits' => 'per_day' ) ); ?>"><?php _e( 'Limit per day', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'limits' => 'per_day' ) ); ?>" id="<?php echo $this->field_id( array( 'limits' => 'per_day' ) ); ?>" value="<?php echo $prefs['limits']['per_day']; ?>" class="form-control" />
				<span class="description"><?php echo $this->core->template_tags_general( __( 'Number of comments per day that grants %_plural%. Use zero for unlimited.', 'mycred' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<div class="hook-instance">
	<h3><?php _e( 'SPAM Comments', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-3 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'spam' => 'creds' ) ); ?>"><?php _e( 'Member', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'spam' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'spam' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['spam']['creds'] ); ?>" class="form-control" />
				<span class="description"><?php _e( 'Use zero to disable.', 'mycred' ); ?></span>
			</div>
		</div>
		<div class="col-lg-3 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'spam' => 'author' ) ); ?>"><?php _e( 'Content Author', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'spam' => 'author' ) ); ?>" id="<?php echo $this->field_id( array( 'spam' => 'author' ) ); ?>" value="<?php echo $this->core->number( $prefs['spam']['author'] ); ?>" class="form-control" />
				<span class="description"><?php _e( 'Use zero to disable.', 'mycred' ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'spam' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'spam' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'spam' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['spam']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'comment' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<div class="hook-instance">
	<h3><?php _e( 'Trashed Comments', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-3 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'trash' => 'creds' ) ); ?>"><?php _e( 'Member', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'trash' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'trash' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['trash']['creds'] ); ?>" class="form-control" />
				<span class="description"><?php _e( 'Use zero to disable.', 'mycred' ); ?></span>
			</div>
		</div>
		<div class="col-lg-3 col-md-6 col-sm-6 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'trash' => 'author' ) ); ?>"><?php _e( 'Content Author', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'trash' => 'author' ) ); ?>" id="<?php echo $this->field_id( array( 'trash' => 'author' ) ); ?>" value="<?php echo $this->core->number( $prefs['trash']['author'] ); ?>" class="form-control" />
				<span class="description"><?php _e( 'Use zero to disable.', 'mycred' ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'trash' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'trash' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'trash' => 'log' ) ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['trash']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'comment' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<?php

		}

		/**
		 * Sanitise Preference
		 * @since 1.1.1
		 * @version 1.1
		 */
		function sanitise_preferences( $data ) {

			$new_data = $data;

			$new_data['limits']['per_post']   = ( ! empty( $data['limits']['per_post'] ) ) ? absint( $data['limits']['per_post'] ) : 0;
			$new_data['limits']['per_day']    = ( ! empty( $data['limits']['per_day'] ) ) ? absint( $data['limits']['per_day'] ) : 0;
			$new_data['limits']['self_reply'] = ( isset( $data['limits']['self_reply'] ) ) ? 1 : 0;

			return $new_data;

		}

	}
endif;

/**
 * Hooks for Clicking on Links
 * @since 1.1
 * @version 1.3
 */
if ( ! class_exists( 'myCRED_Hook_Click_Links' ) ) :
	class myCRED_Hook_Click_Links extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			parent::__construct( array(
				'id'       => 'link_click',
				'defaults' => array(
					'limit_by' => 'none',
					'creds'    => 1,
					'log'      => '%plural% for clicking on link to: %url%'
				)
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 1.1
		 * @version 1.0
		 */
		public function run() {

			if ( ! is_user_logged_in() ) return;

			add_action( 'mycred_register_assets',      array( $this, 'register_script' ) );
			add_action( 'mycred_front_enqueue_footer', array( $this, 'enqueue_footer' ) );
			add_filter( 'mycred_parse_tags_link',      array( $this, 'parse_custom_tags' ), 10, 2 );

			if ( isset( $_POST['action'] ) && $_POST['action'] == 'mycred-click-points' && isset( $_POST['token'] ) && wp_verify_nonce( $_POST['token'], 'mycred-link-points' ) ){
				$this->ajax_call_link_points();
			}

		}

		/**
		 * Customize Limit Options
		 * @since 1.1
		 * @version 1.0
		 */
		public function custom_limit() {

			return array(
				'none' => __( 'No limit', 'mycred' ),
				'url'  => __( 'Once for each unique URL', 'mycred' ),
				'id'   => __( 'Once for each unique link id', 'mycred' )
			);

		}

		/**
		 * Parse Custom Tags in Log
		 * @since 1.1
		 * @version 1.1.1
		 */
		public function parse_custom_tags( $content, $log_entry ) {

			$data    = maybe_unserialize( $log_entry->data );
			$content = str_replace( '%url%', $data['link_url'], $content );
			$content = str_replace( '%id%',  $data['link_id'], $content );

			if ( isset( $data['link_title'] ) )
				$content = str_replace( '%title%',  $data['link_title'], $content );

			return $content;

		}

		/**
		 * Register Script
		 * @since 1.1
		 * @version 1.0
		 */
		public function register_script() {

			global $mycred_link_points;

			$mycred_link_points = false;

			wp_register_script(
				'mycred-link-points',
				plugins_url( 'assets/js/links.js', myCRED_THIS ),
				array( 'jquery' ),
				myCRED_VERSION . '.1',
				true
			);

		}

		/**
		 * WP Fotter
		 * @since 1.1
		 * @version 1.1
		 */
		public function enqueue_footer() {

			global $mycred_link_points;

			if ( $mycred_link_points === true ) {

				global $post;

				wp_localize_script(
					'mycred-link-points',
					'myCREDlink',
					array(
						'ajaxurl' => esc_url( isset( $post->ID ) ? get_permalink( $post->ID ) : home_url( '/' ) ),
						'token'   => wp_create_nonce( 'mycred-link-points' )
					)
				);
				wp_enqueue_script( 'mycred-link-points' );

			}

		}

		/**
		 * Custom Has Entry Check
		 * @since 1.1
		 * @version 1.1.1
		 */
		public function has_entry( $action = '', $reference = '', $user_id = '', $data = '', $type = '' ) {

			global $wpdb;

			if ( $this->prefs['limit_by'] == 'url' ) {
				$reference = urldecode( $reference );
				$string = '%s:8:"link_url";s:' . strlen( $reference ) . ':"' . $reference . '";%';
			}
			elseif ( $this->prefs['limit_by'] == 'id' ) {
				$string = '%s:7:"link_id";s:' . strlen( $reference ) . ':"' . $reference . '";%';
			}
			else return false;

			$sql = "SELECT id FROM {$this->core->log_table} WHERE ref = %s AND user_id = %d AND data LIKE %s AND ctype = %s;";
			$wpdb->get_results( $wpdb->prepare( $sql, $action, $user_id, $string, $this->mycred_type ) );
			if ( $wpdb->num_rows > 0 ) return true;

			return false;

		}

		/**
		 * AJAX Call Handler
		 * @since 1.1
		 * @version 1.5
		 */
		public function ajax_call_link_points() {

			// We must be logged in
			if ( ! is_user_logged_in() ) return;

			// Make sure we only handle our own point type
			if ( ! isset( $_POST['ctype'] ) || $_POST['ctype'] != $this->mycred_type || ! isset( $_POST['url'] ) ) return;

			// Security
			check_ajax_referer( 'mycred-link-points', 'token' );

			// Current User
			$user_id = get_current_user_id();

			if ( mycred_force_singular_session( $user_id, 'mycred-last-linkclick' ) )
				wp_send_json( 101 );

			// Check if user should be excluded
			if ( $this->core->exclude_user( $user_id ) ) wp_send_json( 200 );

			// Token
			if ( ! isset( $_POST['key'] ) ) wp_send_json( 300 );
			$token = mycred_verify_token( $_POST['key'], 4 );
			if ( $token === false ) wp_send_json( 305 );

			list ( $amount, $point_type, $id, $url ) = $token;
			if ( $amount == '' || $point_type == '' || $id == '' || $url == '' ) wp_send_json( 310 );

			// Make sure the token is not abused
			if ( $url != urlencode( $_POST['url'] ) ) wp_send_json( 315 );

			// Bail now if this was not intenteded for this type
			if ( $point_type != $this->mycred_type ) return;

			// Amount
			if ( $amount == 0 )
				$amount = $this->prefs['creds'];
			else
				$amount = $this->core->number( $amount );

			if ( $amount == 0 || $amount == $this->core->zero() ) wp_send_json( 400 );

			$data = array(
				'ref_type'   => 'link',
				'link_url'   => esc_url_raw( $_POST['url'] ),
				'link_id'    => $id,
				'link_title' => ( isset( $_POST['etitle'] ) ) ? sanitize_text_field( $_POST['etitle'] ) : ''
			);

			// Limits
			if ( $this->prefs['limit_by'] == 'url' ) {
				if ( $this->has_clicked( $user_id, 'link_url', $data['link_url'] ) ) wp_send_json( 600 );
			}
			elseif ( $this->prefs['limit_by'] == 'id' ) {
				if ( $this->has_clicked( $user_id, 'link_id', $data['link_id'] ) ) wp_send_json( 700 );
			}

			// Execute
			$this->core->add_creds(
				'link_click',
				$user_id,
				$amount,
				$this->prefs['log'],
				'',
				$data,
				$point_type
			);

			// Report the good news
			wp_send_json( 'done' );

		}

		/**
		 * Has Clicked
		 * Checks if a user has received points for a link based on either
		 * an ID or URL.
		 * @since 1.3.3.1
		 * @version 1.0
		 */
		public function has_clicked( $user_id = NULL, $by = '', $check = '' ) {

			global $wpdb;

			$rows  = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$this->core->log_table} WHERE ref = %s AND user_id = %d AND ctype = %s", 'link_click', $user_id, $this->mycred_type ) );
			if ( count( $rows ) == 0 ) return false;

			$reply = false;
			foreach ( $rows as $row ) {

				$data = maybe_unserialize( $row->data );
				if ( ! is_array( $data ) || ! isset( $data[ $by ] ) ) continue;

				if ( $data[ $by ] == $check ) {
					$reply = true;
					break;
				}

			}

			return $reply;

		}

		/**
		 * Preference for Link Click Hook
		 * @since 1.1
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<div class="row">
		<div class="col-lg-4 col-md-4 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'creds' ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'creds' ); ?>" id="<?php echo $this->field_id( 'creds' ); ?>" value="<?php echo $this->core->number( $prefs['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-8 col-md-8 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'log' ); ?>"><?php _e( 'Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'log' ); ?>" id="<?php echo $this->field_id( 'log' ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general', 'user' ), '%url%, %title% or %id%' ); ?></span>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for=""><?php _e( 'Limits', 'mycred' ); ?></label>
<?php 

			add_filter( 'mycred_hook_impose_limits', array( $this, 'custom_limit' ) );
			$this->impose_limits_dropdown( 'limit_by', false );

?>
			</div>
		</div>
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label><?php _e( 'Available Shortcode', 'mycred' ); ?></label>
				<p class="form-control-static"><a href="http://codex.mycred.me/shortcodes/mycred_link/" target="_blank">[mycred_link]</a></p>
			</div>
		</div>
	</div>
</div>
<?php

		}

	}
endif;

/**
 * Hooks for Viewing Videos
 * @since 1.2
 * @version 1.1
 */
if ( ! class_exists( 'myCRED_Hook_Video_Views' ) ) :
	class myCRED_Hook_Video_Views extends myCRED_Hook {

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			parent::__construct( array(
				'id'       => 'video_view',
				'defaults' => array(
					'creds'    => 1,
					'log'      => '%plural% for viewing video',
					'logic'    => 'play',
					'interval' => '',
					'leniency' => 10
				)
			), $hook_prefs, $type );

		}

		/**
		 * Run
		 * @since 1.2
		 * @version 1.0.2
		 */
		public function run() {

			global $mycred_video_points;

			$mycred_video_points = array();

			add_action( 'mycred_front_enqueue', array( $this, 'register_script' ), 90 );
			add_action( 'template_redirect',    array( $this, 'maybe_reward_points' ), 5 );
			add_action( 'wp_footer',            array( $this, 'footer' ) );

		}

		/**
		 * Register Script
		 * @since 1.2
		 * @version 1.1.2
		 */
		public function register_script() {

			if ( ! is_user_logged_in() ) return;

			wp_register_script(
				'mycred-video-points',
				plugins_url( 'assets/js/video.js', myCRED_THIS ),
				array( 'jquery' ),
				myCRED_VERSION . '.1',
				true
			);

			global $post;

			wp_localize_script(
				'mycred-video-points',
				'myCRED_Video',
				array(
					'ajaxurl'          => esc_url( ( isset( $post->ID ) ) ? get_permalink( $post->ID ) : home_url( '/' ) ),
					'token'            => wp_create_nonce( 'mycred-video-points' ),
					'default_interval' => abs( $this->prefs['interval']*1000 ),
					'default_logic'    => $this->prefs['logic']
				)
			);
			wp_enqueue_script( 'mycred-video-points' );

			wp_register_script(
				'mycred-video-youtube',
				plugins_url( 'assets/js/youtube.js', myCRED_THIS ),
				array( 'jquery' ),
				myCRED_VERSION . '.1',
				true
			);

		}

		/**
		 * Load Scripts in Footer
		 * @since 1.3.3
		 * @version 1.0.1
		 */
		public function footer() {

			global $mycred_video_points;

			// If youtube videos are used
			if ( in_array( 'youtube', (array) $mycred_video_points ) )
				wp_enqueue_script( 'mycred-video-youtube' );

		}

		/**
		 * Maybe Reward Points
		 * Point rewards are moved from admin-ajax.php to the front end.
		 * @since 1.7.6
		 * @version 1.0.1
		 */
		public function maybe_reward_points( $template ) {

			if ( is_user_logged_in() ) {

				if ( isset( $_POST['action'] ) && $_POST['action'] == 'mycred-viewing-videos' && isset( $_POST['setup'] ) && isset( $_POST['type'] ) && $_POST['type'] == $this->mycred_type && isset( $_POST['token'] ) && wp_verify_nonce( $_POST['token'], 'mycred-video-points' ) ) {

					$user_id  = get_current_user_id();
					if ( $this->core->exclude_user( $user_id ) ) wp_send_json_error();

					$key      = sanitize_text_field( $_POST['setup'] );
					$setup    = mycred_verify_token( $key, 5 );
					if ( $setup === false ) wp_send_json_error();

					list ( $source, $video_id, $amount, $logic, $interval ) = $setup;

					// Required
					if ( empty( $source ) || empty( $video_id ) )  wp_send_json_error();

					// Prep
					$amount   = $this->core->number( $amount );
					$interval = abs( $interval / 1000 );

					// Get playback details
					$actions  = sanitize_text_field( $_POST['video_a'] );
					$seconds  = absint( $_POST['video_b'] );
					$duration = absint( $_POST['video_c'] );
					$state    = absint( $_POST['video_d'] );

					// Apply Leniency
					$leniency = $duration * ( $this->prefs['leniency'] / 100 );
					$leniency = floor( $leniency );
					$watched  = $seconds + $leniency;

					$status   = 'silence';

					switch ( $logic ) {

						// Award points when video starts
						case 'play' :

							if ( $state == 1 ) {

								if ( ! $this->has_entry( 'watching_video', '', $user_id, $video_id, $this->mycred_type ) ) {

									// Execute
									$this->core->add_creds(
										'watching_video',
										$user_id,
										$amount,
										$this->prefs['log'],
										0,
										$video_id,
										$this->mycred_type
									);

									$status = 'added';

								}
								else {

									$status = 'max';

								}

							}

						break;

						// Award points when video is viewed in full
						case 'full' :

							// Check for skipping or if we watched more (with leniency) then the video length
							if ( ! preg_match( '/22/', $actions, $matches ) || $watched >= $duration ) {

								if ( $state == 0 ) {

									if ( ! $this->has_entry( 'watching_video', '', $user_id, $video_id, $this->mycred_type ) ) {

										// Execute
										$this->core->add_creds(
											'watching_video',
											$user_id,
											$amount,
											$this->prefs['log'],
											0,
											$video_id,
											$this->mycred_type
										);

										$status = 'added';

									}
									else {
										$status = 'max';
									}

								}

							}

						break;

						// Award points in intervals
						case 'interval' :

							// The maximum points a video can earn you
							$num_intervals = floor( $duration / $interval );
							$max           = abs( $num_intervals * $amount );
							$users_log     = $this->get_users_video_log( $video_id, $user_id );

							// Film is playing and we just started
							if ( $state == 1 && $users_log === NULL ) {

								// Add points without using mycred_add to prevent
								// notifications from being sent as this amount will change.
								$this->core->update_users_balance( $user_id, $amount );

								$this->core->add_to_log(
									'watching_video',
									$user_id,
									$amount,
									$this->prefs['log'],
									0,
									$video_id,
									$this->mycred_type
								);

								$status = 'added';

							}

							// Film is playing and we have not yet reached maximum on this movie
							elseif ( $state == 1 && isset( $users_log->creds ) && $users_log->creds+$amount <= $max ) {

								$this->update_creds( $users_log->id, $user_id, $users_log->creds+$amount );
								$this->core->update_users_balance( $user_id, $amount );
								$amount = $users_log->creds+$amount;

								$status = 'added';

							}

							// Film has ended and we have not reached maximum
							elseif ( $state == 0 && isset( $users_log->creds ) && $users_log->creds+$amount <= $max ) {

								$this->update_creds( $users_log->id, $user_id, $users_log->creds+$amount );
								$this->core->update_users_balance( $user_id, $amount );
								$amount = $users_log->creds+$amount;

								$status = 'max';

								// If enabled, add notification
								if ( function_exists( 'mycred_add_new_notice' ) ) {

									if ( $amount < 0 )
										$color = '<';
									else
										$color = '>';

									$message = str_replace( '%amount%', $amount, $this->prefs['template'] );
									if ( ! empty( $message ) )
										mycred_add_new_notice( array( 'user_id' => $user_id, 'message' => $message, 'color' => $color ) );

								}

							}

						break;
					}

					wp_send_json( array(
						'status'   => $status,
						'video_id' => $video_id,
						'amount'   => $amount,
						'duration' => $duration,
						'seconds'  => $seconds,
						'watched'  => $watched,
						'actions'  => $actions,
						'state'    => $state,
						'logic'    => $logic,
						'interval' => $interval
					) );

				}

			}

		}

		/**
		 * Get Users Video Log
		 * Returns the log for a given video id.
		 * @since 1.2
		 * @version 1.0.1
		 */
		public function get_users_video_log( $video_id, $user_id ) {

			global $wpdb;

			return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->core->log_table} WHERE user_id = %d AND data = %s AND ctype = %s;", $user_id, $video_id, $this->mycred_type ) );

		}

		/**
		 * Update Points
		 * @since 1.2
		 * @version 1.1
		 */
		public function update_creds( $row_id, $user_id, $amount ) {

			// Prep format
			if ( ! isset( $this->core->format['decimals'] ) )
				$decimals = $this->core->core['format']['decimals'];

			else
				$decimals = $this->core->format['decimals'];

			if ( $decimals > 0 )
				$format = '%f';

			else
				$format = '%d';

			$amount = $this->core->number( $amount );

			global $wpdb;

			$wpdb->update(
				$this->core->log_table,
				array( 'creds' => $amount ),
				array( 'id'    => $row_id ),
				array( $format ),
				array( '%d' )
			);

		}

		/**
		 * Preference for Viewing Videos
		 * @since 1.2
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<div class="row">
		<div class="col-lg-4 col-md-4 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'creds' ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'creds' ); ?>" id="<?php echo $this->field_id( 'creds' ); ?>" value="<?php echo $this->core->number( $prefs['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-8 col-md-8 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( 'log' ); ?>"><?php _e( 'Log Template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( 'log' ); ?>" id="<?php echo $this->field_id( 'log' ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['log'] ); ?>" class="form-control" />
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-7 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'logic' => 'play' ) ); ?>"><?php _e( 'Award Logic', 'mycred' ); ?></label>
				<div class="checkbox">
					<label for="<?php echo $this->field_id( array( 'logic' => 'play' ) ); ?>"><input type="radio" name="<?php echo $this->field_name( 'logic' ); ?>" id="<?php echo $this->field_id( array( 'logic' => 'play' ) ); ?>"<?php checked( $prefs['logic'], 'play' ); ?> value="play" class="toggle-hook-option" /> <?php _e( 'Play - As soon as video starts playing.', 'mycred' ); ?></label>
				</div>
				<div class="checkbox">
					<label for="<?php echo $this->field_id( array( 'logic' => 'full' ) ); ?>"><input type="radio" name="<?php echo $this->field_name( 'logic' ); ?>" id="<?php echo $this->field_id( array( 'logic' => 'full' ) ); ?>"<?php checked( $prefs['logic'], 'full' ); ?> value="full" class="toggle-hook-option" /> <?php _e( 'Full - First when the entire video has played.', 'mycred' ); ?></label>
				</div>
				<div class="checkbox">
					<label for="<?php echo $this->field_id( array( 'logic' => 'interval' ) ); ?>"><input type="radio" name="<?php echo $this->field_name( 'logic' ); ?>" id="<?php echo $this->field_id( array( 'logic' => 'interval' ) ); ?>"<?php checked( $prefs['logic'], 'interval' ); ?> value="interval" class="toggle-hook-option" /> <?php echo $this->core->template_tags_general( __( 'Interval - For each x number of seconds watched.', 'mycred' ) ); ?></label>
				</div>
			</div>
		</div>
		<div class="col-lg-5 col-md-6 col-sm-12 col-xs-12">
			<div id="<?php echo $this->field_id( array( 'logic-option-interval' ) ); ?>"<?php if ( $prefs['logic'] != 'interval' ) echo ' style="display: none;"';?>>
				<div class="form-group">
					<label for="<?php echo $this->field_id( 'interval' ); ?>"><?php _e( 'Intervals', 'mycred' ); ?></label>
					<input type="text" name="<?php echo $this->field_name( 'interval' ); ?>" id="<?php echo $this->field_id( 'interval' ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['interval'] ); ?>" class="form-control" />
					<span class="description"><?php printf( __( 'The number of seconds a user must watch in order to get %s.', 'mycred' ), $this->core->plural() ); ?></span>
				</div>
			</div>
			<div id="<?php echo $this->field_id( array( 'logic-option-full' ) ); ?>"<?php if ( $prefs['logic'] != 'full' ) echo ' style="display: none;"';?>>
				<div class="form-group">
					<label for="<?php echo $this->field_id( 'leniency' ); ?>"><?php _e( 'Leniency', 'mycred' ); ?></label>
					<input type="text" name="<?php echo $this->field_name( 'leniency' ); ?>" id="<?php echo $this->field_id( 'leniency' ); ?>" placeholder="<?php _e( 'required', 'mycred' ); ?>" value="<?php echo esc_attr( $prefs['leniency'] ); ?>" class="form-control" />
					<span class="description"><?php _e( 'Do not set this value to zero! A lot of thing can happen while a user watches a movie and sometimes a few seconds can drop of the counter due to buffering or play back errors.', 'mycred' ); ?></span>
				</div>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-12 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label><?php _e( 'Available Shortcode', 'mycred' ); ?></label>
				<p class="form-control-static"><a href="http://codex.mycred.me/shortcodes/mycred_video/" target="_blank">[mycred_video]</a></p>
			</div>
		</div>
	</div>
</div>
<script type="text/javascript">
jQuery(function($){

	$( '#sidebar-active .toggle-hook-option' ).change(function(){

		if ( $(this).val() == 'interval' ) {
			$( '#<?php echo $this->field_id( array( 'logic-option-interval' ) ); ?>' ).show();
			$( '#<?php echo $this->field_id( array( 'logic-option-full' ) ); ?>' ).hide();
		}
		else if ( $(this).val() == 'full' ) {
			$( '#<?php echo $this->field_id( array( 'logic-option-full' ) ); ?>' ).show();
			$( '#<?php echo $this->field_id( array( 'logic-option-interval' ) ); ?>' ).hide();
		}
		else {
			$( '#<?php echo $this->field_id( array( 'logic-option-full' ) ); ?>' ).hide();
			$( '#<?php echo $this->field_id( array( 'logic-option-interval' ) ); ?>' ).hide();
		}

	});

});
</script>
<?php

		}

	}
endif;

/**
 * Hook for affiliations
 * @since 1.4
 * @version 1.3.1
 */
if ( ! class_exists( 'myCRED_Hook_Affiliate' ) ) :
	class myCRED_Hook_Affiliate extends myCRED_Hook {

		public $ref_key  = '';
		public $limit_by = array();

		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type = MYCRED_DEFAULT_TYPE_KEY ) {

			parent::__construct( array(
				'id'       => 'affiliate',
				'defaults' => array(
					'visit'    => array(
						'creds'    => 1,
						'log'      => '%plural% for referring a visitor',
						'limit'    => 1,
						'limit_by' => 'total'
					),
					'signup'    => array(
						'creds'    => 10,
						'log'      => '%plural% for referring a new member',
						'limit'    => 1,
						'limit_by' => 'total'
					),
					'setup' => array(
						'links'    => 'username',
						'IP'       => 1
					),
					'buddypress' => array(
						'profile'  => 0,
						'priority' => 10,
						'title'    => __( 'Affiliate Program', 'mycred' ),
						'desc'     => ''
					)
				)
			), $hook_prefs, $type );

			// Let others play with the limit by
			$this->limit_by = apply_filters( 'mycred_affiliate_limit_by', array(
				'total' => __( 'Total', 'mycred' ),
				'daily' => __( 'Per Day', 'mycred' )
			), $this );

			// Let others play with the ref key
			$this->ref_key = apply_filters( 'mycred_affiliate_key', 'mref', $this );

			add_filter( 'mycred_parse_log_entry_signup_referral', array( $this, 'parse_log_entry' ), 10, 2 );

		}

		/**
		 * Run
		 * @since 1.4
		 * @version 1.2.1
		 */
		public function run() {

			// Insert into BuddyPress profile
			if ( function_exists( 'bp_is_active' ) && bp_is_active( 'xprofile' ) && $this->prefs['buddypress']['profile'] )
				add_action( 'bp_after_profile_loop_content', array( $this, 'buddypress_profile' ), $this->prefs['buddypress']['priority'] );

			// Hook into user activation
			if ( function_exists( 'buddypress' ) && apply_filters( 'bp_core_signup_send_activation_key', true ) )
				add_action( 'mycred_bp_user_activated', array( $this, 'verified_signup' ) );

			// Register Shortcodes
			add_filter( 'mycred_affiliate_link_' . $this->mycred_type, array( $this, 'shortcode_affiliate_link' ), 10, 2 );
			add_filter( 'mycred_affiliate_id_' . $this->mycred_type,   array( $this, 'shortcode_affiliate_id' ), 10, 2 );

			add_filter( 'mycred_referral_keys', array( $this, 'add_key' ) );

			// Logged in users do not get points
			if ( is_user_logged_in() && apply_filters( 'mycred_affiliate_allow_members', false ) === false ) return;

			// Points for visits
			if ( $this->prefs['visit']['creds'] != 0 || $this->prefs['signup']['creds'] != 0 )
				add_action( 'mycred_referred_visit', array( $this, 'site_visits' ) );

			// Points for signups
			if ( $this->prefs['signup']['creds'] != 0 )
				add_action( 'mycred_referred_signup', array( $this, 'site_signup' ) );

		}

		/**
		 * Parse Log Entry
		 * Add support for user related template tags in signup referrals.
		 * @since 1.4
		 * @version 1.0
		 */
		public function parse_log_entry( $content, $entry ) {

			$user_id = absint( $entry->ref_id );
			return $this->core->template_tags_user( $content, $user_id );

		}

		/**
		 * Add Referral Key
		 * @since 1.5.3
		 * @version 1.0
		 */
		public function add_key( $keys ) {

			if ( ! isset( $_GET[ $this->ref_key ] ) || isset( $_COOKIE[ 'mycred_ref' . $this->mycred_type ] ) ) return $keys;

			if ( ! in_array( $this->ref_key, $keys ) )
				$keys[] = $this->ref_key;

			return $keys;

		}

		/**
		 * Shortcode: Affiliate Link
		 * Appends the current users affiliate link to either a given
		 * URL or if not set, the current URL. If user is not logged in,
		 * the set URL is returned. If this is not set, the shortcode
		 * will return an empty string.
		 * @since 1.4
		 * @version 1.1
		 */
		public function shortcode_affiliate_link( $content = '', $atts ) {

			extract( shortcode_atts( array(
				'url'     => 0,
				'user_id' => '',
				'post_id' => ''
			), $atts ) );

			if ( ! is_user_logged_in() && $user_id == '' )
				return $url;

			if ( $user_id == '' )
				$user_id = get_current_user_id();

			if ( $post_id != '' )
				$url = get_permalink( $post_id );

			return $this->get_ref_link( $user_id, $url );

		}

		/**
		 * Shortcode: Affiliate ID
		 * Returns the current users affiliate ID. Returns an empty
		 * string if the user is not logged in.
		 * @since 1.4
		 * @version 1.1
		 */
		public function shortcode_affiliate_id( $content = '', $atts ) {

			extract( shortcode_atts( array(
				'user_id' => ''
			), $atts ) );

			if ( ! is_user_logged_in() && $user_id == '' )
				$ref_id = '';

			else {

				if ( $user_id == '' )
					$user_id = get_current_user_id();

				$ref_id = $this->get_ref_id( $user_id );

			}

			return apply_filters( 'mycred_affiliate_id', $ref_id, $atts, $this );

		}

		/**
		 * BuddyPress Profile
		 * @since 1.4
		 * @version 1.1
		 */
		public function buddypress_profile() {

			// Prep
			$output  = '';
			$user_id = bp_displayed_user_id();

			// Check for exclusion
			if ( $this->core->exclude_user( $user_id ) ) return;

			// If it is my profile or other members allowed to view eachothers profiles or if we are admins
			if ( bp_is_my_profile() || mycred_is_admin() ) {

				$users_ref_link = $this->get_ref_link( $user_id, home_url( '/' ) );

				$output .= '<div class="bp-widget mycred">';

				// Title if set
				if ( $this->prefs['buddypress']['title'] != '' )
					$output .= '<h4>' . $this->prefs['buddypress']['title'] . '</h4>';

				// Table
				$output .= '<table class="profile-fields">';
				$output .= sprintf( '<tr class="field_1 field_ref_link"><td class="label">%s</td><td>%s</td></tr>', __( 'Link', 'mycred' ), $users_ref_link );

				// Show Visitor referral count
				if ( $this->prefs['visit']['creds'] != 0 )
					$output .= sprintf( '<tr class="field_2 field_ref_count_visit"><td class="label">%s</td><td>%s</td></tr>', __( 'Visitors Referred', 'mycred' ), mycred_count_ref_instances( 'visitor_referral', $user_id, $this->mycred_type ) );

				// Show Signup referral count
				if ( $this->prefs['signup']['creds'] != 0 )
					$output .= sprintf( '<tr class="field_3 field_ref_count_signup"><td class="label">%s</td><td>%s</td></tr>', __( 'Signups Referred', 'mycred' ), mycred_count_ref_instances( 'signup_referral', $user_id, $this->mycred_type ) );

				$output .= '</table>';

				// Description if set
				if ( ! empty( $this->prefs['buddypress']['desc'] ) )
					$output .= wpautop( wptexturize( $this->prefs['buddypress']['desc'] ) );

				$output .= '</div>';
			}

			$output = do_shortcode( $output );
			echo apply_filters( 'mycred_affiliate_bp_profile', $output, $user_id, $users_ref_link, $this );

		}

		/**
		 * Visits
		 * @since 1.4
		 * @version 1.3.1
		 */
		public function site_visits() {

			// Required
			if ( ! isset( $_GET[ $this->ref_key ] ) || empty( $_GET[ $this->ref_key ] ) || isset( $_COOKIE[ 'mycred_ref' . $this->mycred_type ] ) ) return;

			// Attempt to get the user id based on the referral id
			$user_id = $this->get_user_id_from_ref_id( $_GET[ $this->ref_key ] );
			if ( $user_id !== NULL && ! is_user_logged_in() ) {

				// Attempt to get the users IP
				$IP = apply_filters( 'mycred_affiliate_IP', $_SERVER['REMOTE_ADDR'], 'visit', $this );
				if ( $IP != '' && $IP != '0.0.0.0' ) {

					// If referral counts
					if ( $this->ref_counts( $user_id, $IP ) ) {

						// Award
						$this->core->add_creds(
							'visitor_referral',
							$user_id,
							$this->prefs['visit']['creds'],
							$this->prefs['visit']['log'],
							time(),
							$IP,
							$this->mycred_type
						);

						do_action( 'mycred_visitor_referral', $user_id, $IP, $this );

					}

					// Set cookies
					if ( ! headers_sent() ) {

						setcookie( 'mycred_ref' . $this->mycred_type, $_GET[ $this->ref_key ], apply_filters( 'mycred_affiliate_cookie', ( time()+3600*24 ), false, $this ), COOKIEPATH, COOKIE_DOMAIN );

						if ( get_option( 'users_can_register' ) && $this->prefs['signup']['creds'] > 0 )
							setcookie( 'signup_ref' . $this->mycred_type, $_GET[ $this->ref_key ], apply_filters( 'mycred_affiliate_cookie', ( time()+3600*24 ), true, $this ), COOKIEPATH, COOKIE_DOMAIN );

					}

				}

			}

		}

		/**
		 * Signups
		 * @since 1.4
		 * @version 1.2.1
		 */
		public function site_signup( $new_user_id ) {

			// Requirement
			$ref = false;
			$key = '';
			if ( isset( $_COOKIE[ 'signup_ref' . $this->mycred_type ] ) ) {
				$ref = $_COOKIE[ 'signup_ref' . $this->mycred_type ];
				$key = 'signup_ref' . $this->mycred_type;
			}
			elseif ( isset( $_COOKIE[ 'mycred_ref' . $this->mycred_type ] ) ) {
				$ref = $_COOKIE[ 'mycred_ref' . $this->mycred_type ];
				$key = 'mycred_ref' . $this->mycred_type;
			}

			if ( $ref === false ) return;

			// Attempt to get the user id based on the referrer
			$user_id = $this->get_user_id_from_ref_id( $ref );
			if ( $user_id === NULL ) {

				if ( ! headers_sent() )
					setcookie( $key, $ref, time()-3600, COOKIEPATH, COOKIE_DOMAIN );

				return;

			}

			// Delete Cookie
			if ( ! headers_sent() )
				setcookie( $key, $ref, time()-3600, COOKIEPATH, COOKIE_DOMAIN );

			// Attempt to get the users IP
			$IP = apply_filters( 'mycred_affiliate_IP', $_SERVER['REMOTE_ADDR'], 'signup', $this );
			if ( $IP != '' && $IP != '0.0.0.0' ) {

				if ( $this->ref_counts( $user_id, $IP, 'signup' ) ) {

					// Award when users account gets activated
					if ( function_exists( 'buddypress' ) && apply_filters( 'bp_core_signup_send_activation_key', true ) === true ) {
						mycred_add_user_meta( $new_user_id, 'referred_by', '', $user_id );
						mycred_add_user_meta( $new_user_id, 'referred_by_IP', '', $IP );
						mycred_add_user_meta( $new_user_id, 'referred_by_type', '', $this->mycred_type );
					}

					// Award now
					else {

						$this->core->add_creds(
							'signup_referral',
							$user_id,
							$this->prefs['signup']['creds'],
							$this->prefs['signup']['log'],
							$new_user_id,
							$IP,
							$this->mycred_type
						);

						do_action( 'mycred_signup_referral', $user_id, $IP, $new_user_id, $this );

					}

				}

			}

		}

		/**
		 * Verified Signup
		 * If signups need to be verified, award points now.
		 * @since 1.5
		 * @version 1.0
		 */
		public function verified_signup( $user_id ) {

			// Check if there is a referral
			$referred_by    = mycred_get_user_meta( $user_id, 'referred_by', '', true );
			$referred_by_IP = mycred_get_user_meta( $user_id, 'referred_by_IP', '', true );
			$referred_type  = mycred_get_user_meta( $user_id, 'referred_by_type', '', true );

			if ( $referred_by == '' || $referred_by_IP == '' || $this->mycred_type != $referred_type ) return;

			// Award
			$this->core->add_creds(
				'signup_referral',
				$referred_by,
				$this->prefs['signup']['creds'],
				$this->prefs['signup']['log'],
				$user_id,
				$referred_by_IP,
				$this->mycred_type
			);

			do_action( 'mycred_signup_referral', $referred_by, $referred_by_IP, $user_id, $this );

			// Clean up
			mycred_delete_user_meta( $user_id, 'referred_by' );
			mycred_delete_user_meta( $user_id, 'referred_by_IP' );
			mycred_delete_user_meta( $user_id, 'referred_by_type' );

		}

		/**
		 * Get Ref Link
		 * Returns a given users referral id with optional url appended.
		 * @since 1.4
		 * @version 1.0.1
		 */
		public function get_ref_link( $user_id = '', $url = '' ) {

			// User ID is required
			if ( empty( $user_id ) || $user_id === 0 ) return '';

			// Get Ref ID
			$ref_id = $this->get_ref_id( $user_id );
			if ( $ref_id === NULL ) return '';

			// Appent to specific URL
			if ( ! empty( $url ) )
				$link = add_query_arg( array( $this->ref_key => $ref_id ), $url );

			// Append to current URL
			else
				$link = add_query_arg( array( $this->ref_key => $ref_id ) );

			return apply_filters( 'mycred_affiliate_get_ref_link', esc_url( $link ), $user_id, $url, $this );

		}

		/**
		 * Get Ref ID
		 * Returns a given users referral ID.
		 * @since 1.4
		 * @version 1.1
		 */
		public function get_ref_id( $user_id ) {

			$ref_id = NULL;

			// Link format
			switch ( $this->prefs['setup']['links'] ) {

				case 'username' :

					$user = get_userdata( $user_id );
					if ( isset( $user->user_login ) ) $ref_id = urlencode( $user->user_login );

				break;

				case 'numeric' :

					$id = mycred_get_user_meta( $user_id, 'mycred_affiliate_link', '' );
					if ( ! is_numeric( $id ) ) {

						$counter = absint( get_option( 'mycred_affiliate_counter', 0 ) );
						$number  = $counter+1;

						update_option( 'mycred_affiliate_counter', $number );
						mycred_update_user_meta( $user_id, 'mycred_affiliate_link', '', $number );

						$ref_id = $number;

					}
					else {
						$ref_id = $id;
					}

				break;

			}

			return apply_filters( 'mycred_affiliate_get_ref_id', $ref_id, $user_id, $this );

		}

		/**
		 * Get User ID from Ref ID
		 * @since 1.4
		 * @version 1.0.1
		 */
		public function get_user_id_from_ref_id( $string = '' ) {

			global $wpdb;

			$user_id = NULL;

			switch ( $this->prefs['setup']['links'] ) {

				case 'username' :

					$ref_id  = sanitize_text_field( urldecode( $string ) );
					$user    = get_user_by( 'login', $ref_id );
					if ( isset( $user->ID ) )
						$user_id = $user->ID;

				break;

				case 'numeric' :

					$referral_id_key = mycred_get_meta_key( 'mycred_affiliate_link' );
					$ref_id          = absint( $string );
					$user_id         = $wpdb->get_var( $wpdb->prepare( "SELECT user_id FROM {$wpdb->usermeta} WHERE meta_key = %s AND meta_value = %d;", $referral_id_key, $ref_id ) );

				break;

			}

			// Make sure if the referring user is excluded we do not do anything
			if ( $user_id !== NULL && $this->core->exclude_user( $user_id ) )
				$user_id = NULL;

			return apply_filters( 'mycred_affiliate_get_user_id', $user_id, $string, $this );

		}

		/**
		 * Ref Counts
		 * Checks to see if this referral counts.
		 * @since 1.4
		 * @version 1.2
		 */
		public function ref_counts( $user_id, $IP = '', $instance = 'visit' ) {

			global $wpdb;

			// Prep
			$reply = true;

			if ( $instance == 'signup' )
				$ref = 'signup_referral';
			else
				$ref = 'visitor_referral';

			// We start by enforcing the global IP rule
			if ( $this->prefs['setup']['IP'] > 0 ) {

				// Count the occurence of this IP
				$count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$this->core->log_table} WHERE ref = %s AND data = %s AND ctype = %s;", $ref, $IP, $this->mycred_type ) );

				if ( $count !== NULL && $count >= $this->prefs['setup']['IP'] )
					$reply = false;

			}

			// If reply is still true we check limit
			if ( $reply !== false && $this->over_hook_limit( $instance, $ref, $user_id ) )
				$reply = false;

			return apply_filters( 'mycred_affiliate_ref_counts', $reply, $this );

		}

		/**
		 * Preference for Affiliate Hook
		 * @since 1.4
		 * @version 1.1
		 */
		public function preferences() {

			$prefs = $this->prefs;

?>
<div class="hook-instance">
	<h3><?php _e( 'Referring Visitors', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'visit' => 'creds' ) ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'visit' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'visit' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['visit']['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'visit', 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'visit', 'limit' ) ), $this->field_id( array( 'visit', 'limit' ) ), $prefs['visit']['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'visit' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'visit' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'visit' => 'log' ) ); ?>" value="<?php echo esc_attr( $prefs['visit']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general' ) ); ?></span>
			</div>
		</div>
	</div>
</div>
<div class="hook-instance">
	<h3><?php _e( 'Referring Signups', 'mycred' ); ?></h3>

	<?php if ( get_option( 'users_can_register' ) ) : ?>

	<div class="row">
		<div class="col-lg-2 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'signup' => 'creds' ) ); ?>"><?php echo $this->core->plural(); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'signup' => 'creds' ) ); ?>" id="<?php echo $this->field_id( array( 'signup' => 'creds' ) ); ?>" value="<?php echo $this->core->number( $prefs['signup']['creds'] ); ?>" class="form-control" />
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'signup', 'limit' ) ); ?>"><?php _e( 'Limit', 'mycred' ); ?></label>
				<?php echo $this->hook_limit_setting( $this->field_name( array( 'signup', 'limit' ) ), $this->field_id( array( 'signup', 'limit' ) ), $prefs['signup']['limit'] ); ?>
			</div>
		</div>
		<div class="col-lg-6 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'signup' => 'log' ) ); ?>"><?php _e( 'Log template', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'signup' => 'log' ) ); ?>" id="<?php echo $this->field_id( array( 'signup' => 'log' ) ); ?>" value="<?php echo esc_attr( $prefs['signup']['log'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->available_template_tags( array( 'general' ) ); ?></span>
			</div>
		</div>
	</div>

	<?php else : ?>

	<div class="row">
		<div class="col-lg-12 col-md-12 col-sm-12 col-xs-12">
			<p>Registrations are disabled.</p>
			<input type="hidden" name="<?php echo $this->field_name( array( 'signup' => 'creds' ) ); ?>" value="<?php echo esc_attr( $this->defaults['signup']['creds'] ); ?>" />
			<input type="hidden" name="<?php echo $this->field_name( array( 'signup' => 'limit' ) ); ?>" value="<?php echo esc_attr( $this->defaults['signup']['limit'] ); ?>" />
			<input type="hidden" name="<?php echo $this->field_name( array( 'signup' => 'log' ) ); ?>" value="<?php echo esc_attr( $this->defaults['signup']['log'] ); ?>" />
		</div>
	</div>

	<?php endif; ?>

</div>
<div class="hook-instance">
	<h3><?php _e( 'Referral Links', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'setup' => 'links' ) ); ?>-numeric"><input type="radio" name="<?php echo $this->field_name( array( 'setup' => 'links' ) ); ?>" id="<?php echo $this->field_id( array( 'setup' => 'links' ) ); ?>-numeric" <?php checked( $prefs['setup']['links'], 'numeric' ); ?> value="numeric" /> <?php _e( 'Assign numeric referral IDs to each user.', 'mycred' ); ?></label>
				<span class="description"><?php printf( '%s: %s', __( 'Example', 'mycred' ), esc_url( add_query_arg( array( $this->ref_key => 1 ), home_url( '/' ) ) ) ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'setup' => 'links' ) ); ?>-username"><input type="radio" name="<?php echo $this->field_name( array( 'setup' => 'links' ) ); ?>" id="<?php echo $this->field_id( array( 'setup' => 'links' ) ); ?>-username" <?php checked( $prefs['setup']['links'], 'username' ); ?> value="username" /> <?php _e( 'Assign usernames as IDs for each user.', 'mycred' ); ?></label>
				<span class="description"><?php printf( '%s: %s', __( 'Example', 'mycred' ), esc_url( add_query_arg( array( $this->ref_key => 'john+doe' ), home_url( '/' ) ) ) ); ?></span>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'setup' => 'IP' ) ); ?>"><?php _e( 'IP Limit', 'mycred' ); ?></label>
				<input type="text" name="<?php echo $this->field_name( array( 'setup' => 'IP' ) ); ?>" id="<?php echo $this->field_id( array( 'setup' => 'IP' ) ); ?>" value="<?php echo absint( $prefs['setup']['IP'] ); ?>" class="form-control" />
				<span class="description"><?php echo $this->core->template_tags_general( __( 'The number of times each IP address grants %_plural%. Use zero for unlimited.', 'mycred' ) ); ?></span>
			</div>
		</div>
		<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label><?php _e( 'Available Shortcodes', 'mycred' ); ?></label>
				<p class="form-control-static"><a href="http://codex.mycred.me/shortcodes/mycred_affiliate_link/" target="_blank">[mycred_affiliate_link]</a>, <a href="http://codex.mycred.me/shortcodes/mycred_affiliate_id/" target="_blank">[mycred_affiliate_id]</a></p>
			</div>
		</div>
	</div>
</div>
<?php if ( function_exists( 'bp_is_active' ) && bp_is_active( 'xprofile' ) ) : ?>
<div class="hook-instance">
	<h3><?php _e( 'BuddyPress Profile', 'mycred' ); ?></h3>
	<div class="row">
		<div class="col-lg-12 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<div class="checkbox">
					<label for="<?php echo $this->field_id( array( 'buddypress' => 'profile' ) ); ?>"><input type="checkbox" name="<?php echo $this->field_name( array( 'buddypress' => 'profile' ) ); ?>" id="<?php echo $this->field_id( array( 'buddypress' => 'profile' ) ); ?>"<?php checked( $prefs['buddypress']['profile'], 1 ); ?> value="1" /> <?php _e( 'Insert referral link in users profiles', 'mycred' ); ?></label>
				</div>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-8 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label><?php _e( 'Title', 'mycred' ); ?></label><br />
				<input type="text" name="<?php echo $this->field_name( array( 'buddypress' => 'title' ) ); ?>" id="<?php echo $this->field_id( array( 'buddypress' => 'title' ) ); ?>" value="<?php echo esc_attr( $prefs['buddypress']['title'] ); ?>" class="form-control" />
				<span class="description"><?php _e( 'Leave empty to hide.', 'mycred' ); ?></span>
			</div>
		</div>
		<div class="col-lg-4 col-md-6 col-sm-12 col-xs-12">
			<div class="form-group">
				<label><?php _e( 'Profile Positioning', 'mycred' ); ?></label><br />
				<input type="text" name="<?php echo $this->field_name( array( 'buddypress' => 'priority' ) ); ?>" id="<?php echo $this->field_id( array( 'buddypress' => 'priority' ) ); ?>" value="<?php echo absint( $prefs['buddypress']['priority'] ); ?>" class="form-control" />
				<span class="description"><?php _e( 'You can move around the referral link on your users profile by changing the position. Increase to move up, decrease to move down.', 'mycred' ); ?></span>
			</div>
		</div>
	</div>
	<div class="row">
		<div class="col-lg-12 col-md-12 col-sm-12 col-xs-12">
			<div class="form-group">
				<label for="<?php echo $this->field_id( array( 'buddypress' => 'desc' ) ); ?>"><?php _e( 'Description', 'mycred' ); ?></label>
				<span class="description"><?php _e( 'Optional description to insert under the link.', 'mycred' ); ?></span>
				<textarea name="<?php echo $this->field_name( array( 'buddypress' => 'desc' ) ); ?>" id="<?php echo $this->field_id( array( 'buddypress' => 'desc' ) ); ?>" class="form-control" rows="5" cols="30"><?php echo esc_attr( $prefs['buddypress']['desc'] ); ?></textarea>
			</div>
		</div>
	</div>
</div>
<?php else : ?>
<input type="hidden" name="<?php echo $this->field_name( array( 'buddypress' => 'profile' ) ); ?>" value="<?php echo esc_attr( $this->defaults['buddypress']['profile'] ); ?>" />
<input type="hidden" name="<?php echo $this->field_name( array( 'buddypress' => 'title' ) ); ?>" value="<?php echo esc_attr( $this->defaults['buddypress']['title'] ); ?>" />
<input type="hidden" name="<?php echo $this->field_name( array( 'buddypress' => 'desc' ) ); ?>" value="<?php echo esc_attr( $this->defaults['buddypress']['desc'] ); ?>" />
<input type="hidden" name="<?php echo $this->field_name( array( 'buddypress' => 'priority' ) ); ?>" value="<?php echo esc_attr( $this->defaults['buddypress']['priority'] ); ?>" />
<?php endif; ?>
<?php

			do_action( 'mycred_affiliate_prefs', $prefs, $this );

		}

		/**
		 * Sanitise Preference
		 * @since 1.4
		 * @version 1.1
		 */
		function sanitise_preferences( $data ) {

			$data['buddypress']['profile'] = ( isset( $data['buddypress']['profile'] ) ) ? $data['buddypress']['profile'] : 0;

			if ( isset( $data['visit']['limit'] ) && isset( $data['visit']['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['visit']['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['visit']['limit'] = $limit . '/' . $data['visit']['limit_by'];
				unset( $data['visit']['limit_by'] );
			}

			if ( isset( $data['signup']['limit'] ) && isset( $data['signup']['limit_by'] ) ) {
				$limit = sanitize_text_field( $data['signup']['limit'] );
				if ( $limit == '' ) $limit = 0;
				$data['signup']['limit'] = $limit . '/' . $data['signup']['limit_by'];
				unset( $data['signup']['limit_by'] );
			}

			return apply_filters( 'mycred_affiliate_save_pref', $data );

		}

	}
endif;
