<?php

/**
 * Scratch Card Item
 * Class Scratch_Card
 */
class Scratch_Card
{
    /**
     * @var
     * ID of the Card: Level
     */
    private $id;

    /**
     * @var
     * Text
     */
    private $text;

    /**
     * @var
     * Url of the Background
     */
    private $background;

    /**
     * @var
     * Font Size
     */
    private $fontsize;

    /**
     * @var
     * Text Color in Hex
     */
    private $textColor;

    /**
     * Redirect Link when user Scratch the Card
     * @var
     */
    private $redirect_link;

    /**
     * @var
     * Earned Points
     * If MyCred Is Installed
     */
    private $mycred_points;

    /**
     * @var
     * Is A winning Value
     * Lower priority of this value
     */
    private $is_winning_value;

    /**
     * Generated Picture URL
     * @var
     */
    private $generated_picture_url;

    /**
     * base Path
     * @var
     */
    private $path;

    /**
     * Global Post Content
     * @var
     */
    private static $post_content;

    /**
     * Width
     * @var
     */
    private static $width;

    /**
     * Height
     * @var
     */
    private static $height;

    /**
     * Current Post ID
     * @var
     */
    private static $post_id;

    /**
     * Base Path
     * @var
     */
    private static $base_path;

    /**
     * Foreground
     */
    private static $foreground;

    /**
     * @return mixed
     */
    public function getText()
    {
        return $this->text;
    }

    /**
     * @param mixed $text
     */
    public function setText($text)
    {
        $this->text = $text;
        // Update Image
    }

    /**
     * @return mixed
     */
    public function getBackground()
    {
        return $this->background;
    }

    /**
     * @param mixed $background
     */
    public function setBackground($background)
    {
        $this->background = filter_var($background, FILTER_SANITIZE_URL);
    }

    /**
     * @return mixed
     */
    public function getFontSize()
    {
        return $this->fontsize;
    }

    /**
     * @param mixed $fontsize
     */
    public function setFontSize($fontsize)
    {
        $this->fontsize = filter_var($fontsize, FILTER_SANITIZE_NUMBER_INT);
    }

    /**
     * @return mixed
     */
    public function getTextColor()
    {
        return $this->textColor;
    }

    /**
     * @param mixed $textColor
     * @return mixed
     */
    public function setTextColor($textColor)
    {
        if ($this->validate_color($textColor))
        {
            return $this->textColor = $textColor;
        }

        $this->textColor = '#00000';
    }

    /**
     * @return mixed
     */
    public function getRedirectLink()
    {
        return $this->redirect_link;
    }

    /**
     * @param mixed $redirect_link
     */
    public function setRedirectLink($redirect_link)
    {
        $this->redirect_link = filter_var($redirect_link, FILTER_SANITIZE_URL);
    }

    /**
     * @return mixed
     */
    public function getMycredPoints()
    {
        if (is_null($this->mycred_points))
        {
            return 0.0;
        }

        return $this->mycred_points;
    }

    /**
     * @param mixed $mycred_points
     */
    public function setMycredPoints($mycred_points)
    {
        $this->mycred_points = filter_var($mycred_points, FILTER_SANITIZE_NUMBER_FLOAT);
    }

    /**
     * @return mixed
     */
    public function getIsWinningValue()
    {
        if (is_null($this->is_winning_value))
        {
            return FALSE;
        }
        return ($this->is_winning_value == 'on' ? TRUE : FALSE);
    }

    /**
     * @param mixed $is_winning_value
     */
    public function setIsWinningValue($is_winning_value)
    {
        $this->is_winning_value = $is_winning_value;
    }


    /**
     * Get Generated Image
     */
    public function getGeneratedImage()
    {
        if (empty($this->generated_picture_url))
        {
            return $this->generateImage();
        }

        return $this->generated_picture_url;
    }

    /**
     * @return mixed
     * get the ID
     */
    public function getID()
    {
        return $this->id;
    }

    /**
     * @param $id
     * Set The ID
     */
    public function setID($id)
    {
        $this->id = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
    }

    /**
     * To Array
     * @return array
     */
    public function toArray()
    {
        if (empty($this->getText()) && empty($this->getBackground()))
        {
            return null;
        }

        $arr = array(
            'id'           => $this->getID(),
            'text'         => $this->getText(),
            'background'   => $this->getBackground(),
            'fontSize'     => $this->getFontSize(),
            'textColor'    => $this->getTextColor(),
            'redirectLink' => $this->getRedirectLink(),
            'myCredPoints' => $this->getMycredPoints(),
            'IsWinningValue' => $this->getIsWinningValue()
        );

        if (!is_null($this->generated_picture_url))
        {
            $arr = array_merge($arr, array('generated_url' => $this->generated_picture_url));
        }

        return $arr;
    }

    /**
     * @return string
     * Get a JSON representation of the class
     */
    public function toJson()
    {
        return json_encode($this->toArray());
    }


    /**
     * update the class by a json string
     * @param $json
     */
    public function fromJson($json)
    {
        $array = json_decode($json, true);
        $this->setArray($array);
    }

    /**
     * Update class by an Array
     * @param $args
     */
    public function setArray($args)
    {
        if (isset($args['id']))
        {
            $this->setID($args['id']);
        }

        if (isset($args['text']))
        {
            $this->setText($args['text']);
        }

        if (isset($args['background']))
        {
            $this->setBackground($args['background']);
        }

        if (isset($args['fontSize']))
        {
            $this->setFontSize($args['fontSize']);
        }

        if (isset($args['textColor']))
        {
            $this->setTextColor($args['textColor']);
        }

        if (isset($args['redirectLink']))
        {
            $this->setRedirectLink($args['redirectLink']);
        }

        if (isset($args['myCredPoints']))
        {
            $this->setMycredPoints($args['myCredPoints']);
        }

        if (isset($args['IsWinningValue']))
        {
            $this->setIsWinningValue($args['IsWinningValue']);
        }
    }

    /**
     * Scratch_Card constructor.
     * @param mixed $args
     */
    public function __construct($args)
    {
        if (is_array($args))
        {
            $this->setArray($args);
        }
        else
        {
            $this->fromJson($args);
        }

    }


    /**
     * Set Post Content Globally
     * @param $content
     */
    public static function setPostContent($content)
    {
        static::$post_content = $content;
    }

    /**
     * set Height
     * @param $height
     */
    public static function setHeight($height)
    {
        static::$height = $height;
    }

    /**
     * Get Height
     * @return mixed
     */
    public static function getHeight()
    {
        if (is_null(static::$height))
        {
            return 400;
        }

        return static::$height;
    }

    /**
     * Set Width
     * @param $width
     */
    public static function setWidth($width)
    {
        static::$width = $width;
    }

    /**
     * Get Width
     * @return mixed
     */
    public static function getWidth()
    {
        if (is_null(static::$width))
        {
            return 600;
        }

        return static::$width;
    }

    /**
     * Set Post ID
     * @param $id
     */
    public static function setPostID($id)
    {
        self::$post_id = $id;
    }

    /**
     * set Base Path
     * @param $path
     */
    public static function setBasePath($path)
    {
        self::$base_path = $path;
    }

    /**
     * Set Foreground Path
     * @param $path
     */
    public static function setForeground($path)
    {
        self::$foreground = filter_var($path, FILTER_SANITIZE_URL);
    }


    /**
     * Validate color in hexadecimal format
     * @param $color
     * @return bool
       */
    private function validate_color($color)
    {
        if ($color === '') return FALSE;
        if (!preg_match('/^#[a-f0-9]{6}$/i', $color))
            return FALSE;
        return TRUE;
    }

    /**
     * Start Generating Image
     * @param bool $force
     * @return string
     * Generated Image URL
     */
    public function generateImage($force=false)
    {
        if (is_null($this->text))
        {
            return null;
        }

       /*if (is_null($this->text) || (!$force && !empty($this->generated_picture_url)))
        {
            return null;
        }
       */

        if (!empty(static::$post_content))
        {
            $content = static::$post_content . PHP_EOL . $this->getText();
        }
        else
        {
            $content = $this->getText();
        }

        $this->path = wp_upload_dir()['basedir']. static::$base_path;
        $this->path .= '/' . md5(mt_rand(5,12)+$this->getID()). '_Scratch_'.$this->getID().'.png';
        $ImageGen = new ImageGen($this->path, static::getWidth(), static::getHeight(), 'arial.ttf', $content, static::$base_path);
        $ImageGen->setFontSize($this->getFontSize());
        $ImageGen->setFontColor($this->getTextColor());

        if (!empty($this->getBackground()))
        {
            $ImageGen->setBaseImagePath($this->getBackground());
        }

        return $this->generated_picture_url = $ImageGen->getImage(); // Get Generated URL
    }

    /**
     * Destroy Post Generated Image.
     */
    public function Destroy()
    {
        $basepath = '/wp-scratchit/content/post_'.static::$post_id;

        if (!is_null($this->generated_picture_url))
        {
            $segments = parse_url($this->generated_picture_url);
            $path = wp_upload_dir()['basedir'].$segments['path'];
            unlink($path);
        }

    }
}


/**
 * Scratch Cards
 * Class Scratch_CardCollection
 */
class Scratch_CardCollection
{

    /**
     * @var
     * Items
     */
    protected $items;

    /**
     * @var
     *
     */
    protected $count;

    /**
     * @param Scratch_Card $card
     * @return bool
     */
    public function Add(Scratch_Card $card)
    {
        if (!empty($card->getText()) || !empty($card->getBackground()))
        {
            $this->items[] = $card;
            $this->count++;
            return TRUE;
        }

        return FALSE;
    }

    public function AddFromArray(array $cards)
    {
        $inc = false;
        foreach($cards as $id => $card)
        {
            if ($id == 0) $inc = true;
            if ($inc) $id++; 
            $card['id'] = $id;
            $this->Add(new Scratch_Card($card));
        }
    }

    /**
     * @return array
     */
    public function getItems()
    {
        return $this->items;
    }

    /**
     * Scratch_CardCollection constructor.
     *
     */
    public function __construct()
    {
        $this->items = [];
        $this->count = 0;
    }

    /**
     * @return mixed
     */
    public function Count()
    {
        return $this->count;
    }

    /**
     * Export to JSON string
     * @return mixed|string
     */
    public function toJson()
    {
        $data = [];
        foreach($this->items as $item)
        {
            $data[] = $item->toArray();
        }
        return json_encode($data);
    }

    /**
     *
     */
    public function startGeneratingImages()
    {
        foreach($this->items as $item)
        {
            $item->generateImage();
        }
    }

    /**
     * Parse Json and convert items to Cards Items
     * @param $json
     * @return bool
     */
    public function ParseJson($json)
    {
        $items = json_decode($json, true);
        /// (var_dump($items));
        if (!is_null($items))
        {
            $this->AddFromArray($items);
            return true;
        }

        return false;
    }

    /**
     * @param $position
     * @return mixed|null
     */
    public function Peek($position)
    {
        if (isset($this->items[$position - 1]))
        {
            return $this->items[$position - 1];
        }

        return null;
    }

    /**
     * Static function Peek Card from JSON string
     * @param $position
     * @param $json
     * @return mixed|null
     */
    public static function PeekCard($position, $json)
    {
        $collection = new Scratch_CardCollection();
        $collection->ParseJson($json);
        return $collection->Peek($position);
    }

    /**
     * Destroy Card
     * @param $position
     * @return null
     */
    public function Destroy($position)
    {
        if (isset($this->items[$position - 1]))
        {
            return $this->items[$position -1 ]->Destory();
        }

        return null;
    }

    /**
     * Destroy All Cards with Images.
     */
    public static function DestroyAll($post_id)
    {
        $basepath = '/wp-scratchit/content/post_'.$post_id;
        $path = wp_upload_dir()['basedir']. $basepath;


       if ( $handle = @opendir($path) ) {
           while (false !== ($file = readdir($handle))) {
               if ($file == '.' || $file == '..') {
                   continue;
               }
               @unlink(path_join( $path, $file ));
           }
       }

        @rmdir($path);
    }
}

/**
 *
 * Class PostKeyDecorator
 */
class PostKeyDecorator
{
    /**
     * @param $key
     * @return string
     */
    public static function to($key)
    {
        switch ($key)
        {
            case 'wp_scratch_meta_text':
                return 'text';
            case 'wp_scratch_meta_background':
                return 'background';
            case 'wp_scratch_meta_font_size':
                return 'fontSize';
            case 'wp_scratch_meta_text_color':
                return 'textColor';
            case 'wp_scratch_meta_redirect_link':
                return 'redirectLink';
            case 'wp_scratch_meta_mycred_points':
                return 'myCredPoints';
            case 'wp_scratch_meta_is_winning_value':
                return 'IsWinningValue';
        }
    }
}