<?php

/**
 * WordPress ScratchIt
 * http://devplus31.com/wpscratchit
 * DevPlus31
 */

class WPScratch {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Plugin_Name_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the Dashboard and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		$this->plugin_name = 'WP ScratchIt';
		$this->version = '1.5.0';

		$this->load_dependencies();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_shortcode();
		$this->define_custom_posts();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/class-wpscratch-loader.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'admin/class-wpscratch-admin.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'public/class-wpscratch-public.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/class-wpscratch-shortcodeHandler.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/class-wpscratchit-posts.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/class-wpscratch-imageGen.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/class-wpscratch-util.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/Scratch_Card.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/ScratchIt_Entries.php';
		require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/ScratchIt_DashboardWidget.php';

		if (class_exists('myCRED_Hook')) {
			require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/class-wpscratch-mycred.php';	
			WP_Scratch_MyCred::setup_hook();
		}
		else
		{
			require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-wpscratch-logaction.php'; 
			$log = new LogAction();
			$log->run();
		}
		
		$this->loader = new WPScratch_Loader();
	}


	/**
	 * Register all of the hooks related to the dashboard functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new WPScratch_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new WPScratch_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

	}


	private function define_shortcode(){

		$shortcode_handler = new ShortCodeHandler();
		add_shortcode('wp-scratchit' , [$shortcode_handler, 'wp_scratch_show_scratch']);

	}

	private function define_custom_posts()
	{
		add_action('init', function()
		{
            if(!session_id()) {
                session_start();
            }

			$new_post_type = new WPScratchIt_Posts();

		});
	}	


	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Plugin_Name_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
